# MTC Encryption Module

Reusable Laravel helpers for encrypting database attributes and stored files using the framework's `Crypt` facade.

**Requirements**

- PHP 8.2+
- Laravel 10 or 11 (or any project using the matching Illuminate components)

## Installation

1. Register the package in your `composer.json` (path repository or satis entry) and require it:

```json
"require": {
    "mtcmedia/encryption-module": "^1.0"
}
```

2. Publish the configuration (optional):

```bash
php artisan vendor:publish --provider="Mtcmedia\EncryptionModule\EncryptionModuleServiceProvider" --tag=config
```

This creates `config/encryption-module.php`.

## OptionalEncrypted Cast

Attach the cast to any Eloquent attribute that should be transparently encrypted at rest. The package registers the same legacy class path (`App\Casts\OptionalEncrypted`) so existing models do not require changes:

```php
use Mtcmedia\EncryptionModule\Casts\OptionalEncrypted;

protected $casts = [
    'email' => OptionalEncrypted::class,
    'phone' => OptionalEncrypted::class,
];
```

When `encryption-module.enabled` (or the legacy `encryption.enabled`) is `true`, values are encrypted before storage and decrypted on retrieval. Empty values are left untouched.

## ModelEncryptor Helpers

`Mtcmedia\EncryptionModule\Support\ModelEncryptor` (aliased as `App\Src\Encryption`) provides the same chunked migration logic used in this project:

```php
use Mtcmedia\EncryptionModule\Support\ModelEncryptor;

// Encrypt a single model
ModelEncryptor::encrypt(\App\Models\Member::class);

// Decrypt
ModelEncryptor::decrypt(\App\Models\Member::class);

// Update *_hash fields for searchable attributes
ModelEncryptor::updateHashes($member);
```

Register project-specific models inside your published `config/encryption-module.php` (the package ships with an empty list so nothing is encrypted until you opt in) and call the artisan commands without arguments:

- `php artisan encryption:encrypt`
- `php artisan encryption:decrypt`
- `php artisan encryption:run` (legacy alias, iterates configured models)
- `php artisan decryption:run` (legacy alias)

Both commands accept an optional `model` argument and `--chunk=` override.

## File Encryption

`Mtcmedia\EncryptionModule\Services\FileEncrypter` wraps `Storage` and `Crypt` to secure uploaded files:

```php
use Mtcmedia\EncryptionModule\Services\FileEncrypter;

public function store(UploadedFile $file, FileEncrypter $encrypter)
{
    $meta = $encrypter->storeEncryptedFile($file, 'local', 'patient-files');

    // $meta['path'] holds the encrypted blob, e.g. patient-files/12345.pdf.enc
}

public function download(string $path, FileEncrypter $encrypter)
{
    return $encrypter->streamDownload('local', $path, 'report.pdf');
}
```

The suffix appended to encrypted blobs defaults to `.enc` and can be changed via config (`file.suffix`).

## Configuration

| Key | Description |
| --- | --- |
| `enabled` | Toggles encryption globally (defaults to `ENCRYPTION_ENABLED` env). |
| `models` | Array of model class names used by the artisan commands. |
| `chunk_size` | Row chunk size for model migrations. |
| `hash_suffix` | Column suffix for deterministic hashes (defaults to `_hash`). |
| `file.suffix` | Suffix appended to encrypted files. |
| `file.disk` | Optional default filesystem disk used by `FileEncrypter`. |

## Legacy Compatibility

The module honours the legacy `config('encryption.enabled')` flag when `encryption-module.enabled` is missing, making migration painless for older projects.
