<?php

namespace Mtcmedia\EncryptionModule\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Config;
use Mtcmedia\EncryptionModule\Support\ModelEncryptor;

class DecryptModels extends Command
{
    protected $signature = 'encryption:decrypt {model? : Fully qualified model class name} {--chunk= : Override chunk size}';

    protected $description = 'Decrypt attributes for one or more models that use OptionalEncrypted casts.';

    public function handle(): int
    {
        $chunkSize = $this->option('chunk');
        $chunkSize = $chunkSize !== null ? (int) $chunkSize : null;

        $model = $this->argument('model');
        if ($model) {
            $this->decryptModel($model, $chunkSize);
            return self::SUCCESS;
        }

        $models = Config::get('encryption-module.models', []);
        if (empty($models)) {
            $this->info('No models registered in config/encryption-module.php.');
            return self::SUCCESS;
        }

        foreach ($models as $registeredModel) {
            $this->decryptModel($registeredModel, $chunkSize);
        }

        return self::SUCCESS;
    }

    private function decryptModel(string $model, ?int $chunkSize): void
    {
        if (!class_exists($model)) {
            $this->error("Model {$model} does not exist.");
            return;
        }

        $this->line("Decrypting {$model}...");

        try {
            ModelEncryptor::decrypt($model, $chunkSize);
            $this->info("✓ Completed {$model}");
        } catch (\Throwable $exception) {
            $this->error("Failed decrypting {$model}: {$exception->getMessage()}");
        }
    }
}
