<?php

namespace Mtcmedia\EncryptionModule\Support;

use Illuminate\Contracts\Database\Eloquent\CastsAttributes;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Schema;
use Mtcmedia\EncryptionModule\Casts\OptionalEncrypted;

class ModelEncryptor
{
    /**
     * Encrypt plaintext values for a given model that uses OptionalEncrypted casts.
     */
    public static function encrypt(string $modelClass, ?int $chunkSize = null): void
    {
        if (!self::isEnabled()) {
            return;
        }

        $chunkSize ??= (int) Config::get('encryption-module.chunk_size', 2000);

        /** @var Model $model */
        $model = new $modelClass();
        $table = $model->getTable();
        if (!self::tableExists($table, $model->getConnectionName())) {
            return;
        }
        $casts = $model->getCasts();

        $encryptedColumns = self::extractEncryptedColumns($casts);
        if (empty($encryptedColumns)) {
            return;
        }

        $searchableColumns = self::resolveSearchableColumns($model, $encryptedColumns);

        $selectColumns = array_merge(['id'], $encryptedColumns);

        $query = DB::table($table)->select($selectColumns)->orderBy('id');

        $query->chunkById($chunkSize, function ($rows) use ($table, $encryptedColumns, $searchableColumns) {
            $updates = [];

            foreach ($rows as $row) {
                $rowUpdate = ['id' => $row->id];
                $changed = false;

                foreach ($encryptedColumns as $column) {
                    $value = $row->{$column};

                    if ($value === null || $value === '') {
                        continue;
                    }

                    if (self::looksEncrypted($value)) {
                        continue;
                    }

                    $cipher = Crypt::encryptString($value);
                    $rowUpdate[$column] = $cipher;
                    $changed = true;

                    if (in_array($column, $searchableColumns, true)) {
                        $rowUpdate[self::hashColumnName($column)] = self::makeHash($value);
                    }
                }

                if ($changed) {
                    $updates[] = $rowUpdate;
                }
            }

            self::performUpserts($table, $updates);
        }, 'id');
    }

    /**
     * Decrypt ciphertext values for a given model.
     */
    public static function decrypt(string $modelClass, ?int $chunkSize = null): void
    {
        if (!self::isEnabled()) {
            return;
        }

        $chunkSize ??= (int) Config::get('encryption-module.chunk_size', 2000);

        /** @var Model $model */
        $model = new $modelClass();
        $table = $model->getTable();
        if (!self::tableExists($table, $model->getConnectionName())) {
            return;
        }
        $casts = $model->getCasts();

        $encryptedColumns = self::extractEncryptedColumns($casts);
        if (empty($encryptedColumns)) {
            return;
        }

        $searchableColumns = self::resolveSearchableColumns($model, $encryptedColumns);

        $selectColumns = array_merge(['id'], $encryptedColumns);

        DB::table($table)
            ->select($selectColumns)
            ->orderBy('id')
            ->chunkById($chunkSize, function ($rows) use ($table, $encryptedColumns, $searchableColumns) {
                $updates = [];

                foreach ($rows as $row) {
                    $rowUpdate = ['id' => $row->id];
                    $changed = false;

                    foreach ($encryptedColumns as $column) {
                        $value = $row->{$column};

                        if ($value === null || $value === '') {
                            continue;
                        }

                        if (!self::looksEncrypted($value)) {
                            if (in_array($column, $searchableColumns, true)) {
                                $rowUpdate[self::hashColumnName($column)] = self::makeHash($value);
                                $changed = true;
                            }
                            continue;
                        }

                        try {
                            $plain = Crypt::decryptString($value);
                        } catch (\Throwable) {
                            continue;
                        }

                        $rowUpdate[$column] = $plain;
                        $changed = true;

                        if (in_array($column, $searchableColumns, true)) {
                            $rowUpdate[self::hashColumnName($column)] = self::makeHash($plain);
                        }
                    }

                    if ($changed) {
                        $updates[] = $rowUpdate;
                    }
                }

                self::performUpserts($table, $updates);
            }, 'id');
    }

    /**
     * Update deterministic hashes on a model instance.
     */
    public static function updateHashes(Model $model, ?array $attributes = null): void
    {
        if (!self::isEnabled()) {
            return;
        }

        $attributes ??= (array) ($model::$searchable ?? []);

        foreach ($attributes as $attribute) {
            $value = $model->getAttribute($attribute);
            if ($value === null || $value === '') {
                continue;
            }

            $hashColumn = self::hashColumnName($attribute);
            $model->{$hashColumn} = self::makeHash($value);
        }

        $model->saveQuietly();
    }

    /**
     * Generate a deterministic hash for searchable columns.
     *
     * @param  string  $value
     */
    public static function makeHash($value): string
    {
        $normalized = Str::lower(trim((string) $value));

        return hash('sha512', $normalized . Config::get('app.key'));
    }

    /**
     * Encrypt all models registered in the config.
     */
    public static function encryptConfiguredModels(?int $chunkSize = null): void
    {
        foreach (Config::get('encryption-module.models', []) as $modelClass) {
            self::encrypt($modelClass, $chunkSize);
        }
    }

    /**
     * Decrypt all models registered in the config.
     */
    public static function decryptConfiguredModels(?int $chunkSize = null): void
    {
        foreach (Config::get('encryption-module.models', []) as $modelClass) {
            self::decrypt($modelClass, $chunkSize);
        }
    }

    /**
     * Determine whether the module is globally enabled.
     */
    public static function isEnabled(): bool
    {
        return (bool) Config::get(
            'encryption-module.enabled',
            Config::get('encryption.enabled', true)
        );
    }

    /**
     * @param  array<string, string|class-string<CastsAttributes>>  $casts
     * @return array<int, string>
     */
    private static function extractEncryptedColumns(array $casts): array
    {
        $encrypted = [];

        foreach ($casts as $attribute => $castType) {
            if (is_a($castType, OptionalEncrypted::class, true)) {
                $encrypted[] = $attribute;
            }
        }

        return $encrypted;
    }

    /**
     * Determine which encrypted attributes are searchable.
     *
     * @param  Model  $model
     * @param  array<int, string>  $encryptedColumns
     * @return array<int, string>
     */
    private static function resolveSearchableColumns(Model $model, array $encryptedColumns): array
    {
        if (!property_exists($model, 'searchable')) {
            return [];
        }

        $searchable = Arr::wrap($model::$searchable ?? []);

        return array_values(array_intersect($encryptedColumns, $searchable));
    }

    /**
     * Heuristic to avoid decrypting plaintext values.
     */
    private static function looksEncrypted($value): bool
    {
        if (!is_string($value)) {
            return false;
        }

        if (!preg_match('~^[A-Za-z0-9/+]+=*$~', $value)) {
            return false;
        }

        $json = base64_decode($value, true);
        if ($json === false) {
            return false;
        }

        $payload = json_decode($json, true);
        if (!is_array($payload)) {
            return false;
        }

        return isset($payload['iv'], $payload['value'], $payload['mac']) || isset($payload['tag']);
    }

    /**
     * Perform batched upserts without exceeding placeholder limits.
     *
     * @param  string  $table
     * @param  array<int, array<string, mixed>>  $updates
     */
    private static function performUpserts(string $table, array $updates): void
    {
        if (empty($updates)) {
            return;
        }

        $allColumns = [];
        foreach ($updates as $update) {
            foreach (array_keys($update) as $column) {
                if ($column !== 'id') {
                    $allColumns[$column] = true;
                }
            }
        }

        $updateColumns = array_keys($allColumns);
        $columnsPerRow = 1 + count($updateColumns);
        $maxRowsByPlaceholder = (int) floor(60000 / max(1, $columnsPerRow));
        $chunkSize = min($maxRowsByPlaceholder, 1000);

        for ($offset = 0, $total = count($updates); $offset < $total; $offset += $chunkSize) {
            $slice = array_slice($updates, $offset, $chunkSize);
            DB::table($table)->upsert($slice, ['id'], $updateColumns);
        }
    }

    private static function hashColumnName(string $attribute): string
    {
        $suffix = Config::get('encryption-module.hash_suffix', '_hash');

        return $attribute . $suffix;
    }

    private static function tableExists(string $table, ?string $connection): bool
    {
        $schema = Schema::connection($connection);

        return $schema->hasTable($table);
    }
}
