<?php

namespace Mtc\Events;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Mtc\Basket\Contracts\Purchasable;
use Mtc\Core\Traits\Nodeable;
use Mtc\Money\HasPrices;
use Illuminate\Support\Str;

/**
 * Class Ticket
 * @package Mtc\Tickets
 */
class Event extends Model implements Purchasable
{
    use HasPrices;
    use Nodeable;

    /**
     * Price fields
     *
     * @var string[]
     */
    protected $price_fields = [
        'ticket_price'
    ];
    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'name',
        'published',
        'starts_at',
        'ends_at',
        'tickets_available_from',
        'tickets_available_until',
        'location',
        'category_id',
        'intro_text',
        'description',
        'image',
        'ticket_price',
        'total_ticket_count',
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'published' => 'boolean',
        'starts_at' => 'datetime',
        'ends_at' => 'datetime',
        'tickets_available_from' => 'datetime',
        'tickets_available_until' => 'datetime',
    ];

    /**
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    /**
     * Append when converting to json
     *
     * @var string[]
     */
    protected $appends = [
        'node_id',
    ];

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        self::saved(function (self $event) {
            if (empty($event->getAttribute('slug'))) {
                $event->slug = Str::slug($event->name);
                if (self::query()->where('slug', $event->slug)->exists()) {
                    $event->slug .= '-' . $event->id;
                }
                $event->save();
            }
        });
    }

    /**
     * Category relationship
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function category()
    {
        return $this->belongsTo(EventCategory::class, 'category_id');
    }

    /**
     * Ticket relationship
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function tickets()
    {
        return $this->hasMany(Ticket::class);
    }

    /**
     * Active event
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeActive(Builder $query)
    {
        return $query->where('published', 1);
    }

    /**
     * Upcoming event
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeUpcoming(Builder $query)
    {
        return $query->where('starts_at', '>=', Carbon::now());
    }

    /**
     * Whether shipping is required
     *
     * @return bool|mixed
     */
    public function requiresShipping()
    {
        return false;
    }


    /**
     * Whether item is available for purchase
     *
     * @return bool
     */
    public function isAvailable(): bool
    {
        if ($this->published === false) {
            return false;
        }

        if ($this->available_ticket_count <= 0) {
            return false;
        }

        if ($this->tickets_available_from && $this->tickets_available_from > Carbon::now()) {
            return false;
        }

        if ($this->tickets_available_until && $this->tickets_available_until < Carbon::now()) {
            return false;
        }

        return true;
    }

    /**
     * Basket weight
     *
     * @return int|mixed
     */
    public function getWeight()
    {
        return 0;
    }

    /**
     * Url to view ticket
     *
     * @return mixed|void
     */
    public function getUrl()
    {
        return route('events.show', $this->slug);
    }

    /**
     * Image for usage in basket
     *
     * @return mixed|void
     */
    public function getBasketImage($size = 'mini')
    {
        return $this->getImage($size);
    }

    /**
     * Get the image
     *
     * @param string $size
     * @param bool $with_domain
     * @return string
     */
    public function getImage($size = 'thumbs', $with_domain = false)
    {
        if (empty($this->attributes['image'])) {
            return '';
        }

        $prefix = $with_domain ? Config::get('app.url') : '';

        $sizes = App::make('image_folders');
        return $prefix . '/' . $sizes['event_images'][$size]['path'] . '/' . $this->image;
    }

    /**
     * Price
     *
     * @return mixed
     */
    public function getOriginalPrice()
    {
        return $this->getPrice();
    }

    /**
     * Check if item is discounted - basket
     *
     * @return bool|mixed
     */
    public function hasDiscountedPrice()
    {
        return false;
    }

    /**
     * Price
     *
     * @return mixed
     */
    public function getPrice()
    {
        return $this->ticket_price;
    }

    /**
     * name
     *
     * @return mixed
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Unique identifier for product
     *
     * @return mixed|string
     */
    public function getSku()
    {
        return 'T-' . $this->id;
    }

    /**
     * stock for product
     *
     * @return int|mixed
     */
    public function getStock()
    {
        return $this->available_ticket_count;
    }

    /**
     * Max quantity per purchase
     *
     * @return mixed|void
     */
    public function getMaxQuantity()
    {
        // TODO: long term implementation for ticket amount restrictions
        return $this->available_ticket_count;
    }

    /**
     * Additional basket details
     *
     * @return array
     */
    public function getAttributeFields(): array
    {
        // For tickets with allocated seats
        // this would be used to specify the row / seat
        return [];
    }

    /**
     * Query for searching in admin
     *
     * @param Builder $query
     * @param string $term
     * @return Builder|mixed
     */
    public function scopeSearchByTerm(Builder $query, string $term)
    {
        return $query->where('name', 'like', "%{$term}%");
    }

    /**
     * When searching in admin return result as this
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return collect([
            $this->name,
            $this->price
        ])->filter()
            ->implode(', ');
    }

    /**
     * Get the node_id attribute
     *
     * @return |null
     */
    public function getNodeIdAttribute()
    {
        return $this->node->id ?? null;
    }

    protected function convertToDateTime($value)
    {
        if (is_string($value)) {
            try {
                return Carbon::createFromFormat('d/m/Y H:i', $value);
            } catch (\Exception $exception) {
                return $value;
            }
        }

        return $value;
    }

    /**
     * Mutate date time for event time
     *
     * @param $value
     * @return void
     */
    public function setStartsAtAttribute($value)
    {
        $this->attributes['starts_at'] =  $this->convertToDateTime($value);
    }

    /**
     * Mutate date time for event time
     *
     * @param $value
     * @return void
     */
    public function setEndsAtAttribute($value)
    {
        $this->attributes['ends_at'] =  $this->convertToDateTime($value);
        ;
    }

    /**
     * Mutate date time for ticket availability
     *
     * @param $value
     * @return void
     */
    public function setTicketsAvailableFromAttribute($value)
    {
        $this->attributes['tickets_available_from'] =  $this->convertToDateTime($value);
    }

    /**
     * Mutate date time for ticket availability
     *
     * @param $value
     * @return void
     */
    public function setTicketsAvailableUntilAttribute($value)
    {
        $this->attributes['tickets_available_until'] =  $this->convertToDateTime($value);
    }

    /**
     * Check if ticket purchase is possible
     *
     * @return bool
     */
    public function canPurchaseTickets()
    {
        if ($this->tickets_available_from && $this->tickets_available_from > Carbon::now()) {
            return false;
        }

        if ($this->tickets_available_until && $this->tickets_available_until < Carbon::now()) {
            return false;
        }

        if ($this->ends_at && $this->ends_at < Carbon::now()) {
            return false;
        }

        return true;
    }

    /**
     * Adjust available ticket count if total ticket count changes
     */
    public function checkTicketCountAdjustment()
    {
        if ($this->isDirty('total_ticket_count') == false) {
            return;
        }

        $diff = $this->total_ticket_count - $this->getOriginal('total_ticket_count');

        $this->attributes['available_ticket_count'] += $diff;
    }
}
