<?php

namespace Mtc\Events\Factories;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Mpdf\QrCode\Output\Png;
use Mtc\Events\Ticket;
use Mtc\Orders\Order;
use Mpdf\QrCode\QrCode;
use Mpdf\Mpdf;

/**
 * Class TicketFactory
 *
 * @package Mtc\Tickets\Factories
 */
class TicketFactory
{
    /**
     * Create ticket based on type
     *
     * @param $event_id
     * @param $data
     * @param $type
     * @return Model|Ticket
     */
    public function create($event_id, $data, $type)
    {
        switch ($type) {
            case 'order':
            default:
                $ticket = $this->createFromOrder($event_id, $data);
                break;
        }

        $ticket->ticketEvent->decrement('available_ticket_count');
        $this->generateTicketPdf($ticket);
        return $ticket;
    }

    /**
     * @param $event_id
     * @param Order $order_data
     * @return Ticket|Model
     */
    protected function createFromOrder($event_id, $order_data)
    {
        return Ticket::query()
            ->create([
                'event_id' => $event_id,
                'purchase_type' => 'order',
                'purchase_id' => $order_data->id,
                'reference' => $this->generateTicketReference($event_id),
                'customer_name' => $order_data->customer_name,
                'member_id' => $order_data->member_id,
                'email' => $order_data->email,
                'contact_number' => $order_data->contact_number,
            ]);
    }


    /**
     * Generate a unique string reference for ticket
     *
     * @return string
     */
    protected function generateTicketReference($event_id): string
    {
        $i = 0;
        $prefix = 'T' . str_pad($event_id, 4, 0, STR_PAD_LEFT);
        do {
            $suffix = Str::upper(Str::random(config('events.ticket_suffix_length')));
            $reference = $prefix . '-' . $suffix;
            // max attempts for generating a ticket
            if (++$i > config('events.ticket_suffix_length') * 20) {
                throw new \Exception('Unable to generate a unique code for ticket');
            }
        } while (Ticket::query()->where('reference', $reference)->exists());

        return $reference;
    }

    /**
     * Generate QR code for ticket
     *
     * @param Ticket $ticket
     * @return void
     */
    public function generateTicketQRCode($ticket)
    {
        $qr_code = new QrCode($ticket->reference);
        $output = new Png();
        $data = $output->output($qr_code);
        Storage::put($ticket->getQRCodePath(), $data);
    }

    /**
     * Generate PDF
     *
     * @param Ticket $ticket
     * @throws \Mpdf\MpdfException
     */
    public function generateTicketPdf($ticket)
    {
        if (!$ticket) {
            return;
        }

        $this->generateTicketQRCode($ticket);

        $ticket_html = template('events/tickets/pdf_template.twig', [
            'ticket' => $ticket
        ]);

        $pdf = new Mpdf();
        $pdf->WriteHTML($ticket_html);

        // Since we're saving from pdf instead of storage
        // The directory needs to be created it won't auto-generate
        if (Storage::exists('tickets') === false) {
            Storage::makeDirectory('tickets');
        }
        $pdf->Output(Storage::path($ticket->getTicketFileLocation()));
    }
}
