<?php

namespace Mtc\Events\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Mtc\Core\Facades\Media;
use Mtc\Core\Images\ImageUploader;
use Mtc\Events\Event;
use Mtc\Events\EventCategory;
use Mtc\Events\Http\Requests\SaveEventRequest;

/**
 * Class EventAdminController
 *
 * @package Mtc\Events
 */
class EventAdminController extends Controller
{

    /**
     * List view
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        $events = Event::query()
            ->paginate(25)
            ->appends($request->except('page'));

        $this->page_meta['title'] = 'Manage Events';
        return template('admin/events/index.twig', [
            'page_meta' => $this->page_meta,
            'events' => $events
        ]);
    }

    /**
     * Create view
     *
     * @param Event $event
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create(Event $event)
    {
        $this->page_meta['title'] = 'Create Event';
        return template('admin/events/edit.twig', [
            'page_meta' => $this->page_meta,
            'event' => $event,
            'image_folders' => Media::getFolders('event_images'),
            'categories' => EventCategory::query()->pluck('name', 'id')->prepend('Choose Category', 0),
        ]);
    }

    /**
     * Save event
     *
     * @param SaveEventRequest $request
     * @param Event $event
     * @param ImageUploader $uploader
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(SaveEventRequest $request, Event $event, ImageUploader $uploader)
    {
        if ($request->file('image')) {
            $file_data = $uploader
                ->setFolders(Media::getFolders('event_images'))
                ->uploadImage($request->file('image'));
            if ($file_data) {
                $event->image = $file_data['name'];
            }
        }
        $event->available_ticket_count = $request->input('total_ticket_count');
        $event->fill($request->input())->save();

        return redirect(route('admin.events.edit', [ $event->id ]))->with('success', 'Event Saved');

    }

    /**
     * Edit view
     *
     * @param Event $event
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Event $event)
    {
        $tickets =$event->tickets()
            ->withTrashed()
            ->get();

        $this->page_meta['title'] = 'Edit Event';
        return template('admin/events/edit.twig', [
            'page_meta' => $this->page_meta,
            'event' => $event,
            'tickets' => $tickets,
            'image_folders' => Media::getFolders('event_images'),
            'categories' => EventCategory::query()->pluck('name', 'id')->prepend('Choose Category', 0),
        ]);
    }

    /**
     * Update event data
     *
     * @param SaveEventRequest $request
     * @param Event $event
     * @param ImageUploader $uploader
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(SaveEventRequest $request, Event $event, ImageUploader $uploader)
    {
        if ($request->file('image')) {
            $file_data = $uploader
                ->setFolders(Media::getFolders('event_images'))
                ->uploadImage($request->file('image'));
            if ($file_data) {
                $event->image = $file_data['name'];
            }
        }
        $event->fill($request->input());
        $event->checkTicketCountAdjustment();
        $event->save();

        return back()->with('success', 'Event Updated');
    }

    /**
     * Remove event
     *
     * @param Event $event
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Exception
     */
    public function destroy(Event $event)
    {
        $event->delete();
        $event->tickets()->delete();
        return back()->with('success', 'Event Deleted');
    }
}
