<?php

namespace Mtc\Filter;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\Filter\Contracts\FilterInstance;
use Mtc\Filter\Contracts\FilterObject;
use Mtc\Filter\Contracts\FilterSeoContract;
use Mtc\Filter\Contracts\IsFilter;

class Filter implements FilterInstance
{
    /**
     * @var Request
     */
    protected Request $request;

    /**
     * @var array
     */
    protected array $config;

    /**
     * @var FilterObject
     */
    protected FilterObject $product_handler;

    /**
     * @var array
     */
    protected array $selections = [];

    /**
     * @var Builder
     */
    protected Builder $query;

    /**
     * @var Collection
     */
    protected Collection $filters;

    /**
     * @var Collection
     */
    protected Collection $custom_pattern_filters;

    /**
     * @var string
     */
    protected string $active_sort_option_name;

    /**
     * @var array
     */
    protected array $filter_url_elements;

    /**
     * @var FilterSeoContract
     */
    protected FilterSeoContract $seo;

    /**
     * Filter constructor
     *
     * @param Request $request
     * @param FilterObject $product_handler
     * @param FilterSeoContract $seo
     */
    public function __construct(Request $request, FilterObject $product_handler, FilterSeoContract $seo)
    {
        $this->request = $request;
        $this->seo = $seo;
        $this->config = Config::get('filter');
        $this->product_handler = $product_handler;
        $this->selections = $this->groupSelectionsByType($request->input('selections', []));
        $this->active_sort_option_name = $this->config['default_sort_choice'];
    }

    /**
     * Parse Request to selections and filter data
     *
     * @return array
     */
    public function parseRequest(): array
    {
        $this->decodeRequest();
        $this->run();

        return [
            'results' => $this->getResults(),
            'filters' => $this->getFilterResults(),
            'seo' => $this->getPageSeoData(),
            'selections' => $this->getSelectionList(),
            'sort_options' => $this->getSortOptions(),
            'sort_by' => $this->active_sort_option_name,
            'base_url' => url($this->config['url_entrypoint']),
            'page' => $this->request->input('page', 1),
        ];
    }

    /**
     * Handle Ajax request, perform filtering
     *
     * @return array
     */
    public function handle(): array
    {
        $this->run();

        return [
            'results' => $this->getResults(),
            'filters' => $this->getFilterResults(),
            'selections' => $this->getSelectionsWithNames(),
            'sort_options' => $this->getSortOptions(),
            'url' => $this->getPageUrl(),
            'seo' => $this->getPageSeoData(),
        ];
    }

    /**
     * Get enabled filters
     *
     * @return Collection
     */
    public function getFilters(): Collection
    {
        if (empty($this->filters)) {
            $this->filters = collect($this->config['filters'])
                ->map(fn($filter_class) => App::make($filter_class))
                ->filter(fn($filter) => $filter instanceof IsFilter);
        }

        return $this->filters;
    }

    /**
     * Get enabled filters that use custom pattern
     *
     * @return Collection
     */
    public function getCustomPatternFilters(): Collection
    {
        if (empty($this->custom_pattern_filters)) {
            $this->custom_pattern_filters = $this->getFilters()
                ->filter(fn(IsFilter $filter) => $filter instanceof CustomPatternFilter);
        }

        return $this->custom_pattern_filters;
    }

    /**
     * Get current selections
     *
     * @param string|null $type
     * @return array
     */
    public function getSelections(string $type = null): array
    {
        if ($type === null) {
            return $this->selections;
        }

        return $this->selections[$type] ?? [];
    }

    /**
     * Build the URL of the page that has been viewed
     *
     * @param bool $absolute
     * @return string
     */
    public function getPageUrl(bool $absolute = true): string
    {
        return route('filter_page', [
            'any' => '/' . $this->filterElementsToSlugs($this->selections)->join('/'),
            'page' => $this->request->input('page') > 1 ? $this->request->input('page') : null,
        ], $absolute);
    }

    /**
     * Get the current pagination page of the filter
     *
     * @return int
     */
    public function getCurrentPage(): int
    {
        return $this->request->input('page', 1);
    }

    /**
     * Build URL for array of selections
     *
     * @param array $selections
     * @return string
     */
    public function urlForSelections(array $selections): string
    {
        return route('filter_page', [
            'any' => '/' . $this->filterElementsToSlugs($selections)->join('/'),
        ]);
    }

    /**
     * Create an instance of Filter for type
     *
     * @param string $type
     * @return IsFilter
     */
    public function instantiateFilter(string $type): IsFilter
    {
        return App::make($this->config['filters'][$type]);
    }

    /**
     * Find the filter types that match this model
     *
     * @param Model $model
     * @return Collection
     */
    public function findIndexTypesForModel(Model $model): Collection
    {
        return $this->getFilters()
            ->filter(fn(IsFilter $filter) => $filter->getModel() === get_class($model))
            ->map(fn($filter, $type) => $type);
    }

    /**
     * Decode request by setting selections and values from hard-load of the page
     */
    public function decodeRequest(): self
    {
        $stripped_url_path = str_replace($this->config['url_entrypoint'] . '/', '', $this->request->path());
        $this->filter_url_elements = explode('/', $stripped_url_path);
        $this->matchSortFromUrlElements();
        $this->matchSelectedFilters();
        $this->matchCustomSelections();
        $this->checkForSearchTerm();
        return $this;
    }

    /**
     * Get the selections as a flat list
     * @return Collection
     */
    public function getSelectionList(): Collection
    {
        return collect($this->selections)
            ->flatMap(fn($filter_group, $filter_type) => collect($filter_group)
                ->map(fn($value) => [
                    'type' => $filter_type,
                    'value' => $value,
                ]));
    }

    /**
     * Retrieve a single filter of all supported options
     *
     * @param string $type
     * @return IsFilter|null
     */
    protected function getFilter(string $type): ?IsFilter
    {
        if (empty($this->filters)) {
            $this->getFilters();
        }

        return $this->filters[$type] ?? null;
    }

    /**
     * Run filtering - build query, perform filtering and set ordering
     *
     * @return void
     */
    protected function run(): void
    {
        $this->query = $this->createQuery();
        $this->applyForResults();
        $this->matchSortFromAjaxRequest();
    }

    /**
     * Retrieve filter results based on selections
     *
     * @return array
     */
    protected function getFilterResults(): array
    {
        return $this->getFilters()
            ->map(fn(IsFilter $filter, $name) => $filter->format($this->retrieveSingleFilterResults($filter, $name)))
            ->filter()
            ->toArray();
    }

    /**
     * Get (product) results.
     * Sets the orderBy for the main query
     * Executes the query to retrieve data
     * Formats data to defined response format
     *
     * @return JsonResource
     */
    protected function getResults(): JsonResource
    {
        return $this->product_handler
            ->format($this->product_handler->getResults($this->setSorting(clone $this->query)));
    }

    /**
     * Fetch the list of supported sort options
     * Returned in format of slug => display name
     *
     * @return array
     */
    protected function getSortOptions(): array
    {
        return collect($this->config['sort_options'])
            ->map(fn($sort_class, $sort_name) => __("filter::filter.sort_options.$sort_name"))
            ->toArray();
    }

    /**
     * Find name values for selections
     */
    protected function getSelectionsWithNames()
    {
        $indexed_selections = $this->getFilterIndexForSelections()
            ->toArray();

        $custom_filters = $this->getCustomPatternFilters();
        $custom_selections = collect($this->selections)
            ->filter(fn($selected, $type) => in_array($type, $custom_filters->keys()->toArray()))
            ->flatMap(fn($filter_group, $filter_type) => collect($filter_group)
                ->map(fn($value) => [
                    'filter_id' => $value,
                    'filter_type' => $filter_type,
                    'name' => $custom_filters[$filter_type]->getSelectionName($value),
                ]))
            ->toArray();

        return array_merge($indexed_selections, $custom_selections);
    }

    /**
     * Apply sort to query.
     * Invokes the sort option class and calls its handle method which will set the sort direction
     *
     * @param $query
     * @return mixed
     */
    protected function setSorting($query)
    {
        return App::make($this->config['sort_options'][$this->active_sort_option_name])->handle($query);
    }

    /**
     * Get seo data for current filter page
     *
     * @return array
     */
    protected function getPageSeoData(): array
    {
        return $this->seo->handle($this);
    }

    /**
     * Initialize database query for result filtering
     *
     * @return Builder
     */
    protected function createQuery(): Builder
    {
        return $this->product_handler->createQuery();
    }

    /**
     * Apply filters for results (products)
     */
    protected function applyForResults(): void
    {
        $this->getFilters()
            ->reject(fn(IsFilter $filter, string $type) => empty($this->selections[$type]))
            ->each(fn(IsFilter $filter, string $type) => $filter->applyFilter($this->query, $this->selections[$type]));
    }

    /**
     * Apply filters for other filters.
     * This applies the same  filters as result filtering except for current active filter.
     * It also adds product filter condition
     *
     * @param Builder $query
     * @param string $active_filter
     */
    protected function applyForFilters(Builder $query, string $active_filter): void
    {
        $this->getFilters()
            ->reject(fn(IsFilter $filter, string $type) => empty($this->selections[$type]) || $active_filter === $type)
            ->each(fn(IsFilter $filter, string $type) => $filter->applyFilter($query, $this->selections[$type]));

        $this->product_handler->applyFilter($query);
    }

    /**
     * Retrieve results for single filter
     *
     * @param IsFilter $filter
     * @param string $filter_name
     * @return Collection
     */
    protected function retrieveSingleFilterResults(Isfilter $filter, string $filter_name): Collection
    {
        $limit = in_array($filter_name, $this->request->input('expanded', []), true)
            ? 0
            : $this->config['filter_limit'];

        return $filter->getResults(
            function ($query) use ($filter_name) {
                $this->applyForFilters($query, $filter_name);
            },
            $limit,
            $this->selections[$filter_name] ?? []
        );
    }
    /**
     * Check if search term is in selections
     */
    protected function checkForSearchTerm(): void
    {
        if ($this->request->filled('search-term')) {
            $this->selections['search'][] = $this->request->input('search-term');
        }
    }

    /**
     * Set the sort option name for post requests
     * This is done by finding if url has a part in its path that is a valid sort option
     * If multiple options are set only last one will be used
     */
    protected function matchSortFromUrlElements(): void
    {
        $matched_sort = array_intersect($this->filter_url_elements, array_keys($this->getSortOptions()));
        $this->filter_url_elements = array_diff($this->filter_url_elements, $matched_sort);
        $this->active_sort_option_name = !empty($matched_sort)
            ? array_pop($matched_sort)
            : $this->config['default_sort_choice'];
    }

    /**
     * Primary method for matching selections on filter
     * This uses FilterIndex table which stores unique slugs for all filterable data objects
     * and applies them as a selection when matched
     */
    protected function matchSelectedFilters(): void
    {
        if (empty($this->filter_url_elements)) {
            return;
        }

        $matched = FilterIndex::query()
            ->whereIn('slug', $this->filter_url_elements)
            ->get();

        $this->filter_url_elements = array_diff($this->filter_url_elements, $matched->pluck('slug')->toArray());

        $this->selections = collect($matched)
            ->groupBy('filter_type')
            ->map(fn($group) => collect($group)->pluck('filter_id'))
            ->toArray();
    }

    /**
     * This method attempts to decode non-matched url patterns against filters
     * If filter does not have normal slug based matching it will be passed to this method.
     * This method looks at all filters that use the CustomPatternFilter interface and try to match slugs against them.
     * If an element matches a filter pattern it is applied as a selection and removed from element list
     */
    protected function matchCustomSelections(): void
    {
        if (empty($this->filter_url_elements)) {
            return;
        }

        $custom_filters = $this->getFilters()
            ->filter(fn(IsFilter $filter) => $filter instanceof CustomPatternFilter);

        $matched_filters = collect($this->filter_url_elements)->flip()
            ->map(fn($index, $slug) => $custom_filters
                ->filter(fn(CustomPatternFilter $filter) => $filter->patternMatched($slug))
                ->keys()
                ->first())
            ->filter();

        $this->filter_url_elements = array_diff($this->filter_url_elements, $matched_filters->keys()->toArray());

        $matched_filters->each(function ($filter_type, $slug) use ($custom_filters) {
            $this->selections[$filter_type][] = $custom_filters[$filter_type]->matchSelections($slug);
        });
    }

    /**
     * Set the sort option name for ajax requests
     * This will use the `sort_by` parameter value if it is a valid sort option
     * Alternatively it falls back to default sort option
     */
    protected function matchSortFromAjaxRequest(): void
    {
        $this->active_sort_option_name = array_key_exists($this->request->input('sort_by'), $this->getSortOptions())
            ? $this->request->input('sort_by')
            : $this->config['default_sort_choice'];
    }


    /**
     * Group selections by type
     * @param array $selections
     * @param string $group_by
     * @return array
     */
    protected function groupSelectionsByType(array $selections = [], string $group_by = 'type'): array
    {
        return collect($selections)
            ->groupBy($group_by)
            ->map(fn($group) => collect($group)->pluck('value'))
            ->toArray();
    }

    /**
     * Create a list of all filter elements (selections, ordering, terms etc.) in one list of slugs
     *
     * @param array $selections
     * @return Collection
     */
    protected function filterElementsToSlugs(array $selections): Collection
    {
        $all = collect([]);
        collect($selections)
            ->sortBy(fn($selection, $type) => array_search($type, array_keys($this->config['filters']), true))
            ->each(function ($type_selections, $type) use ($all) {
                $filter = $this->getFilter($type);
                // Unsupported selection
                if ($filter === null) {
                    return;
                }

                collect($type_selections)->each(function ($selection) use ($all, $filter, $type) {
                    $slugs = $this->getFilterIndexForSelections()
                        ->groupBy('filter_type')
                        ->map(fn(Collection $group) => $group->keyBy('filter_id'));

                    if ($filter instanceof CustomPatternFilter) {
                        $all->push($filter->createSlug($selection));
                    } elseif (isset($slugs[$type][$selection]['slug'])) {
                        $all->push($slugs[$type][$selection]['slug']);
                    }
                });
            });

        if ($this->active_sort_option_name !== $this->config['default_sort_choice']) {
            $all->push($this->active_sort_option_name);
        }
        return $all;
    }

    /**
     * @return Builder[]|\Illuminate\Database\Eloquent\Collection|Collection
     */
    protected function getFilterIndexForSelections()
    {
        if (empty($this->index)) {
            $index_query = FilterIndex::query();
            collect($this->selections)
                ->reject(fn($selections, $type) => $this->getCustomPatternFilters()->has($type))
                ->each(function ($selections, $type) use ($index_query) {
                    $index_query->orWhere(function ($selection_query) use ($type, $selections) {
                        $selection_query->where('filter_type', $type)->whereIn('filter_id', $selections);
                    });
                });
            $this->index = $index_query
                ->get();
        }

        return $this->index;
    }
}
