<?php

namespace Mtc\Filter\Contracts;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

abstract class FilterIndexContract extends Model
{
    protected $table = 'filter_index';

    protected $fillable = [
        'slug',
        'filter_type',
        'filter_id',
    ];

    public static function index(string $filter_type, Model $model, IsFilter $filter)
    {
        $type = $filter_type;
        $id = $model->getAttribute($filter->getIdAttribute());
        $slug = Str::slug($filter->modelSlug($model));

        if (self::checkExisting($type, $id, $slug)) {
            $slug = self::uniqueSlug($type, $id, $slug);
        }

        self::query()
            ->updateOrCreate([
                'filter_type' => $type,
                'filter_id' => $id,
            ], [
                'slug' => $slug
            ]);
    }

    protected static function checkExisting(string $type, $id, string $slug): bool
    {
        return self::query()
            ->where('slug', $slug)
            ->where('filter_id', '!=', $id)
            ->where('filter_type', '!=', $type)
            ->exists();
    }

    protected static function uniqueSlug(string $type, $id, string $slug): string
    {
        $slug_with_type = $slug . '-' . $type;
        if (self::checkExisting($type, $id, $slug_with_type) === false) {
            return $slug_with_type;
        }

        $i = 0;
        do {
            $iterated_slug = $slug . '-' . (++$i);

            if ($i > 100) {
                throw new \Exception('Unable to generate unique filter index slug');
            }
        } while (self::checkExisting($type, $id, $iterated_slug));

        return $iterated_slug;
    }
}
