<?php

namespace Mtc\Filter;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\Filter\Contracts\FilterInstance;
use Mtc\Filter\Contracts\FilterObject;
use Mtc\Filter\Contracts\FilterSeoContract;
use Mtc\Filter\Contracts\IsFilter;

class Filter implements FilterInstance
{
    /**
     * @var Request
     */
    protected $request;

    /**
     * @var array
     */
    protected $config;

    /**
     * @var FilterObject
     */
    protected $product_handler;

    /**
     * @var array
     */
    protected $selections = [];

    /**
     * @var Builder
     */
    protected $query;

    /**
     * @var Collection
     */
    protected $filters;

    /**
     * @var string
     */
    protected $active_sort_option_name;

    /**
     * @var array
     */
    protected $filter_url_elements;

    /**
     * @var FilterSeoContract
     */
    protected $seo;

    public function __construct(Request $request, FilterObject $product_handler, FilterSeoContract $seo)
    {
        $this->request = $request;
        $this->seo = $seo;
        $this->config = Config::get('filter');
        $this->product_handler = $product_handler;
        $this->selections = $this->groupSelectionsByType($request->input('selections', []));
    }

    public function parseRequest(): array
    {
        $this->decodeRequest();

        return [
            'seo' => $this->getPageSeoData(),
            'selections' => $this->getSelectionList(),
            'sort_by' => $this->active_sort_option_name,
            'base_url' => url($this->config['url_entrypoint']),
        ];
    }

    public function handle(): array
    {
        $this->run();

        if ($this->request->filled('expand')) {
            return $this->getSingleFilterResults($this->request->input('expand'));
        }

        return [
            'results' => $this->getResults(),
            'filters' => $this->getFilterResults(),
            'sort_options' => $this->getSortOptions(),
            'url' => $this->getPageUrl(),
            'seo' => $this->getPageSeoData(),
        ];
    }

    public function getFilters(): Collection
    {
        if (empty($this->filters)) {
            $this->filters = collect($this->config['filters'])
                ->map(fn($filter_class) => App::make($filter_class))
                ->filter(fn($filter) => $filter instanceof IsFilter);
        }

        return $this->filters;
    }

    public function getSelections(string $type = null): array
    {
        if ($type === null) {
            return $this->selections;
        }

        return $this->selections[$type] ?? [];
    }

    /**
     * Build the URL of the page that has been viewed
     *
     * @return string
     */
    public function getPageUrl(bool $absolute = true): string
    {
        return route('filter_page', [
            'any' => '/' . $this->allFilterElements($this->selections)->join('/'),
            'page' => $this->request->input('page') > 1 ? $this->request->input('page') : null,
        ], $absolute);
    }

    /**
     * Get the current pagination page of the filter
     *
     * @return int
     */
    public function getCurrentPage(): int
    {
        return $this->request->input('page', 1);
    }

    public function urlForSelections(array $selections): string
    {
        return route('filter_page', [
            'any' => '/' . $this->allFilterElements($selections)->join('/'),
        ]);
    }

    /**
     * Retrieve a single filter of all supported options
     *
     * @param string $type
     * @return IsFilter|null
     */
    protected function getFilter(string $type)
    {
        if (empty($this->filters)) {
            $this->getFilters();
        }

        return $this->filters[$type] ?? null;
    }

    protected function run()
    {
        $this->query = $this->createQuery();
        $this->applyForResults();
        $this->matchSortFromAjaxRequest();
    }

    protected function getSingleFilterResults(string $section)
    {
        return $this->getFilters()
            ->filter(function ($filter, $name) use ($section) {
                return $name === $section;
            })
            ->map(fn(IsFilter $filter, $name) => $filter->format($this->retrieveSingleFilterResults($filter, $name,
                true)))
            ->first();
    }

    protected function getFilterResults(): array
    {
        return $this->getFilters()
            ->map(fn(IsFilter $filter, $name) => $filter->format($this->retrieveSingleFilterResults($filter, $name)))
            ->filter()
            ->toArray();
    }

    /**
     * Get (product) results.
     * Sets the orderBy for the main query
     * Executes the query to retrieve data
     * Formats data to defined response format
     *
     * @return JsonResource
     */
    protected function getResults(): JsonResource
    {
        return $this->product_handler
            ->format($this->product_handler->getResults($this->setSorting(clone $this->query)));
    }

    /**
     * Fetch the list of supported sort options
     * Returned in format of slug => display name
     *
     * @return array
     */
    protected function getSortOptions(): array
    {
        return collect($this->config['sort_options'])
            ->map(function ($sort_class, $sort_name) {
                return __("filter::filter.sort_options.$sort_name");
            })->toArray();
    }

    /**
     * Apply sort to query.
     * Invokes the sort option class and calls its handle method which will set the sort direction
     *
     * @param $query
     * @return mixed
     */
    protected function setSorting($query)
    {
        return App::make($this->config['sort_options'][$this->active_sort_option_name])->handle($query);
    }

    protected function getPageSeoData(): array
    {
        return $this->seo->handle($this);
    }

    /**
     * Initialize database query for result filtering
     *
     * @return Builder
     */
    protected function createQuery(): Builder
    {
        return $this->product_handler->createQuery();
    }

    protected function applyForResults()
    {
        $this->getFilters()
            ->reject(function (IsFilter $filter, string $filter_name) {
                return empty($this->selections[$filter_name]);
            })
            ->each(function (IsFilter $filter, string $filter_name) {
                $filter->applyFilter($this->query, $this->selections[$filter_name]);
            });
    }

    protected function applyForFilters($query = null, $active_filter = false)
    {
        $this->getFilters()
            ->reject(function (IsFilter $filter, string $filter_name) use ($active_filter) {
                return empty($this->selections[$filter_name]) || $active_filter === $filter_name;
            })
            ->each(function (IsFilter $filter, $filter_name) use ($query) {
                $filter->applyFilter($query, $this->selections[$filter_name]);
            });

        $this->product_handler->applyFilter($query);
    }

    protected function retrieveSingleFilterResults(Isfilter $filter, $filter_name, bool $all = false)
    {
        $result_limit = $all ? 0 : $this->config['filter_limit'];
        return $filter->getResults(
            function ($query) use ($filter_name) {
                $this->applyForFilters($query, $filter_name);
            },
            $result_limit,
            $this->selections[$filter_name] ?? []
        );
    }

    /**
     * Decode request by setting selections and values from hard-load of the page
     */
    protected function decodeRequest()
    {
        $stripped_url_path = str_replace($this->config['url_entrypoint'] . '/', '', $this->request->path());
        $this->filter_url_elements = explode('/', $stripped_url_path);
        $this->matchSortFromUrlElements();
        $this->matchSelectedFilters();
        $this->matchCustomSelections();
        $this->checkForSearchTerm();
    }

    protected function checkForSearchTerm()
    {
        if ($this->request->filled('search-term')) {
            $this->selections['search'][] = $this->request->input('search-term');
        }
    }

    /**
     * Set the sort option name for post requests
     * This is done by finding if url has a part in its path that is a valid sort option
     * If multiple options are set only last one will be used
     */
    protected function matchSortFromUrlElements()
    {
        $matched_sort = array_intersect($this->filter_url_elements, array_keys($this->getSortOptions()));
        $this->filter_url_elements = array_diff($this->filter_url_elements, $matched_sort);
        $this->active_sort_option_name = !empty($matched_sort)
            ? array_pop($matched_sort)
            : $this->config['default_sort_choice'];
    }

    /**
     * Primary method for matching selections on filter
     * This uses FilterIndex table which stores unique slugs for all filterable data objects
     * and applies them as a selection when matched
     */
    protected function matchSelectedFilters()
    {
        if (empty($this->filter_url_elements)) {
            return;
        }

        $matched = FilterIndex::query()
            ->whereIn('slug', $this->filter_url_elements)
            ->get();

        $this->filter_url_elements = array_diff($this->filter_url_elements, $matched->pluck('slug')->toArray());

        $this->selections = collect($matched)
            ->groupBy('filter_type')
            ->map(fn($group) => collect($group)->pluck('filter_id'))
            ->toArray();
    }

    /**
     * This method attempts to decode non-matched url patterns against filters
     * If filter does not have normal slug based matching it will be passed to this method.
     * This method looks at all filters that use the CustomPatternFilter interface and try to match slugs against them.
     * If an element matches a filter pattern it is applied as a selection and removed from element list
     */
    protected function matchCustomSelections()
    {
        if (empty($this->filter_url_elements)) {
            return;
        }

        $custom_filters = $this->getFilters()
            ->filter(function (IsFilter $filter) {
                return $filter instanceof CustomPatternFilter;
            });

        $matched_filters = collect($this->filter_url_elements)->flip()
            ->map(function ($index, $slug) use ($custom_filters) {
                return $custom_filters
                    ->filter(function (CustomPatternFilter $filter) use ($slug) {
                        return $filter->patternMatched($slug);
                    })
                    ->keys()
                    ->first();
            })
            ->filter();

        $this->filter_url_elements = array_diff($this->filter_url_elements, $matched_filters->keys()->toArray());

        $matched_filters->each(function ($filter_type, $slug) use ($custom_filters) {
            $this->selections[$filter_type][] = $custom_filters[$filter_type]->matchSelections($slug);
        });
    }

    /**
     * Set the sort option name for ajax requests
     * This will use the `sort_by` parameter value if it is a valid sort option
     * Alternatively it falls back to default sort option
     */
    protected function matchSortFromAjaxRequest(): void
    {
        $this->active_sort_option_name = array_key_exists($this->request->input('sort_by'), $this->getSortOptions())
            ? $this->request->input('sort_by')
            : $this->config['default_sort_choice'];
    }

    protected function getSelectionList(): Collection
    {
        return collect($this->selections)
            ->map(function ($filter_group, $filter_type) {
                return collect($filter_group)
                    ->map(function ($value) use ($filter_type) {
                        return [
                            'type' => $filter_type,
                            'value' => $value,
                        ];
                    });
            })
            ->flatten(1);
    }

    protected function groupSelectionsByType($selections = [], $group_by = 'type'): array
    {
        return collect($selections)
            ->groupBy('type')
            ->map(fn($group) => collect($group)->pluck('value'))
            ->toArray();
    }

    protected function allFilterElements(array $selections): Collection
    {
        $all = collect([]);
        collect($selections)
            ->sortBy(function ($selection, $type) {
                return array_search($type, array_keys($this->config['filters']));
            })
            ->each(function ($type_selections, $type) use ($all) {
                $filter = $this->getFilter($type);
                // Unsupported selection
                if ($filter === null) {
                    return;
                }

                collect($type_selections)->each(function ($selection) use ($all, $filter, $type) {
                    $slugs = $this->getFilterIndex();

                    if ($filter instanceof CustomPatternFilter) {
                        $all->push($filter->createSlug($selection));
                    } elseif (isset($slugs[$type][$selection]['slug'])) {
                        $all->push($slugs[$type][$selection]['slug']);
                    }
                });
            });

        if ($this->active_sort_option_name !== $this->config['default_sort_choice']) {
            $all->push($this->active_sort_option_name);
        }
        return $all;
    }

    protected function getFilterIndex()
    {
        if (empty($this->index)) {
            $this->index = FilterIndex::query()
                ->where([])
                ->get()
                ->groupBy('filter_type')
                ->map(function (Collection $group) {
                    return $group->keyBy('filter_id');
                });
        }

        return $this->index;
    }
}
