<?php

namespace Mtc\Filter\Filters;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Mtc\Filter\Contracts\IsFilter;
use Mtc\Shop\Brand;
use Mtc\Shop\Category;
use Mtc\Shop\Item;

class CategoryFilter extends IsFilter
{

    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('categories', function (Builder $query) use ($selection) {
            $query->whereIn('categories.id', $selection)
                ->active();
        });
    }

    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return Category::query()
            ->active()
            ->whereHas('items', function ($item_query) use ($product_filtering) {
                return $product_filtering($item_query);
            })
            ->where('sub_id', $this->getParentCategory($selections))
            ->addSelect([
                'result_count' => Item::query()
                    ->selectRaw('count(*)')
                    ->join('items_categories', 'item_id', '=', 'items.id')
                    // TODO Active category check
                    ->where(function ($item_query) use ($product_filtering) {
                        $product_filtering($item_query);
                    })
                    ->whereRaw(DB::raw('categories.id = `cat_id`')),

            ])
            ->orderByDesc('result_count')
            ->when($limit > 0, function ($query) use ($limit) {
                $query->limit($limit);
            })
            ->get();
    }

    public function getModel(): string
    {
        return Category::class;
    }

    public function title(): string
    {
        return 'Category';
    }

    public function modelSlug(Model $model): string
    {
        return $model->name;
    }

    public function uiComponentType(): string
    {
        return 'tree-filter';
    }

    /**
     * Determine the display level by setting which parent should be used for results
     *
     * @param array $selections
     * @return int
     */
    protected function getParentCategory(array $selections)
    {
        // No selections - top level
        if (empty($selections)) {
            return 0;
        }

        $last_category = array_pop($selections);
        $has_children = Category::query()
            ->where('sub_id', $last_category)
            ->exists();

        // If current level has children then parent = last selection
        // Alternatively show parent of the last selection to show siblings of last selection
        return $has_children
            ? $last_category
            : (Category::query()->find($last_category)->sub_id ?? 0);
    }
}