<?php

namespace Mtc\Filter\Filters;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\Filter\Contracts\IsFilter;

class SearchTermFilter extends IsFilter implements CustomPatternFilter
{
    protected const TERM_PREFIX = 'q-';

    public function applyFilter($query, array $selection = [])
    {
        collect($selection)->each(function ($search_term) use ($query) {
            $escaped = trim(DB::getPdo()->quote($search_term), "'");
            $query->whereRaw(DB::raw("MATCH(`items`.`name`) AGAINST('*$escaped*' IN BOOLEAN MODE)"));
        });

    }

    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return collect([]);
    }

    public function matchSearchTerms(array $request_elements = []): array
    {
        return collect($request_elements)
            ->filter(function ($element) {
                return Str::startsWith($element, self::TERM_PREFIX);
            })
            ->map(function ($search_term) {
                return [
                    'type' => 'search',
                    'value' => str_replace(self::TERM_PREFIX, '', $search_term)
                ];
            })->toArray();
    }

    public function title(): string
    {
        return 'Search';
    }

    public function patternMatched(string $selection): bool
    {
        return Str::startsWith($selection, 'q-');
    }

    public function matchSelections(string $selection)
    {
        return str_replace([self::TERM_PREFIX, '-'], ['', ' '], $selection);
    }

    public function uiComponentType(): string
    {
        return '';
    }

    /**
     * @param string $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return self::TERM_PREFIX . Str::slug($selection);
    }

    public function getModel(): string
    {
        return '';
    }

    public function modelSlug(Model $model): string
    {
        return '';
    }
}