# Product Filter system

This package provides a filter that can be used for creating a product listing page. It is the key element of 
creating a user journey to find the product they are looking for. 

## Installation

Install via composer
```bash
composer require mtcmedia/filter

# then
php artisan install:components
```

This tool primarily is built to work with mtc's ecommmerce system, but can be adapted to alternate data objects.


## Configuration

The system has been built to allow flexible management of elements in system. All major elements of the 
filter process will have ability to be configured or overwritten via extension.

## Filters

By default system does utilise following filtering options:

* Category
* Brand
* Size
* Colour
* Price
* Search term

These filters are defined in `config/filter.php` as `filters` array. If behaviour of any filter needs to change
the class can be updated in config with the new version that implements correct functionality. If additional 
filters are needed, they can be created via the following command and the 
_newly created class then needs to be added to config file_.  

```bash
php artisan make:product-filter MyFilterName
```

Filter creation can be customised to allow for some specific use cases:

* when creating for custom field, add `--custom-field=db_field`
* when url element is not a simple slug but pattern (e.g. price-from-X-to-Y), add `--custom-pattern`

```bash
php artisan make:product-filter ScreenSize --custom-pattern 
php artisan make:product-filter KeyboardLayout --custom-field=keyboard_layout 
php artisan make:product-filter RegionFilter --custom-field=region --custom-pattern 
```

More information about how filters work [can be found here](docs/filters.md)

## Sort options 

System does allow for various sorting options on results matching the criteria. By default system has:

* Latest Products (New Sdditions)
* Price - Descending
* Price - Ascending
* Alphabetical (A to Z)

Any additional sort choices can be added in `sort_options` section of the filter config. If user does not specify
a specific sort method via UI the `default_sort_choice` config value will be used to sort results.

If additional sort options are needed, they can be created via the following command and the 
_newly created class then needs to be added to config file_.

```bash
php artisan make:product-sort MySortOption
```
Each sort option must be a `Mtc\Filter\Contracts\IsSortOption` implementation.
The interface is simple as it only requires the sort handling to be performed (applying sorting to
the query object). Sort method name is built using language files - `__('filter::filter.sort_options.{sort-option-key})`.

## Result display

Results are queried to database and retrieved based of the main product filtering class.
Class is specified in `filter_object` class. It is imperative to have the results in this 
object returned as an optimised response (i.e. resource of only necessary fields) to ensure
that no unnecessary data is transferred/loaded during filtering process. 

## Display Limits

By Default pagination contains 15 results per page on display result. This can be adjusted based 
on design aspects keeping in mind that the number should be something that will work well with
full page display (usually 3 results per row) and mobile results.

For filters each section will display only few results (amount controlled by `filter_limit` value)
on initial load. If there are additional results to load the system will add load more options link
that will allow loading in additional choices. This allows having a lower page load time and 
less content on page before diving into results. By default system will use the product count 
to sort the order of top results to display. Alternate sorting method (like alphabetical) can be specified
via `filter_limit_type` value. Currently, this support basic sort options, however  enhanced options
like sorting most popular selections (tracking filter click-trough) and trending filters
(tracking click-through over time for filters) are welcome additions.

## SEO

Seo is managed by `Mtc\Filter\Contracts\FilterSeoContract` interface. Default implementation is `Mtc\Filter\FilterSeo`. 
Its primary method is `public function handle(FilterInstance $filter): array;` method which converts the filter data to 
seo information for the page.

To allow specifying page metadata a matching mechanism pipeline is implemented that allows specifying what data should be returned.
Seo Matching pipeline runs through all supported mechanisms to check if the page/filter matches their conditions and 
if it does return the seo data of that mechanism. Only the top mechanism is used. This allows supporting aspects like
matching exact url seo data, then falling back to approximate url, then seo defaults and only then system default.

## Request Lifecycle

Filter uses 2 primary entry-points - hard page load and ajax load.

**Hard page load**

On hard page load filter performs a simple workflow:

* `FilterController::index()` triggers `Filter::parseRequest()`
* `parseRequest()` decodes the request and returns details about the page for template rendering
  * request uri is split in parts by `/` separator and stored into `filter_url_elements`
  * `filter_url_elements` are matched against `sort` options, selected `filters` with standard naming, custom filters with non-standard terms and search terms 
* if inaccessible item flag exists in request, inaccessible item is added to data rendered on template
* `filter/index.twig` template gets rendered 
* template renders `<product-filter></product-filter` Vue Component
* Vue component calls ajax request

**Ajax page load**

Ajax page load does the heavy lifting of the filter functionality

* `FilterController::show()` triggers `Filter::handle()`
* `handle()` triggers `$filter->run()` and then returns all relevant data as json (filters, results, selections, seo, sort options, and the new page url)
* `run()` creates the product query, applies filters to this query and sets sort option on query
* `getResults()` performs result query on the product and formats using the specified product formatting option
* `getFilterResults()` runs through all filters and builds up the query to perform
* `getSelectionsWithNames()` returns all current selections with their friendly display names
* `getSortOptions()` returns all supported sort options
* `getPageUrl()` returns url for page with current selections
* `getPageSeoData()` returns seo data (page title/description/banner)


## UI structure 

UI is driven by Vue + Vuex store.

By default, components are organized as follows:

```vue
<product-filter>
    <filter-header></filter-header>
    <filter-sorting></filter-sorting>
    <filter-sidebar>
      <filter-selections></filter-selections>
      <!-- Normally elements like tree-filter, checkbox-filter and price-range-filter -->
      <component v-for="filter in filters"></component>
    </filter-sidebar>
    <filter-results>
        <filter-result-card v-for="result in results"></filter-result-card>
        <filter-pagination></filter-pagination>
    </filter-results>
</product-filter>
```

The underlying data storage and manipulation of the filter is done through `product_filter_store.js` 
which registers Vuex module `productFilter`. Vuex is used so data loading can be done once per page
and all components on the system can receive data without having to emit events or pass 
properties throughout the structure.

## Filter management / admin

Filter provides 2 elements of management - url slug management and sorting of some filters.

### URL slug management.

Filter uses `FilterIndex` model to create unique url slugs for filterable elements where possible. These are stored
in `filter_index` table. Slugs are auto-generated upon creation and will follow the pattern of creation:

* try to use slug of the model (which is returned by `Filter::modelSlug($model)`) as the slug (conversion to slug happens after returning value).
* if this value is taken then system will attempt to create it in format of `{slug}-{type}` - e.g. `mens-category`
* if also slug with type is taken then it will run a loop to append number at the end `toys-2`, `toys-3` for 100 iterations until a unique value is found. 

Slug is generated only upon first creation of index. It will not auto-modify afterwards. It is possible to manage 
the slug value from the admin page (normally `/admin/filter`) where all registered slugs are displayed by category.
Editing the slug value is done by clicking on a particular slug to open its editing field and then updating it.

It is not possible to modify custom pattern filters from UI as they do not have a fixed format. 

### Filter option ordering

If `Filter::getResults()` is built to use `FilterIndex` model for loading options, options can be made sortable from admin.
This is not fully automatic as code needs to be set to `orderBy('order')` field on the `FilterIndex` model and the filter
object needs to return true for `public function allowIndexOrdering(): bool` method. By default this is available only for 
Size and Colour filters.

When this is made available, admin area will have drag and drop implementation for the filter and will allow saving the order of entries.
