<?php

namespace Mtc\Filter\Filters;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\Filter\Contracts\IsFilter;

class SearchTermFilter extends IsFilter implements CustomPatternFilter
{
    /**
     * Slug prefix
     */
    protected const TERM_PREFIX = 'q-';

    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        collect($selection)->each(function ($search_term) use ($query) {
            $escaped = trim(DB::getPdo()->quote($search_term), "'");
            $query->whereRaw(DB::raw("MATCH(`items`.`name`) AGAINST('*$escaped*' IN BOOLEAN MODE)"));
        });
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return collect([]);
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return '';
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return __('filter::filter.labels.search');
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Specify UI component type used for this filter
     *
     * @return string
     */
    public function uiComponentType(): string
    {
        return '';
    }

    /**
     * Specify whether model should be re-indexed in the index
     * @return bool
     */
    public function reindexModel(): bool
    {
        return false;
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return Str::startsWith($selection, 'q-');
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        return str_replace([self::TERM_PREFIX, '-'], ['', ' '], $selection);
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return self::TERM_PREFIX . Str::slug($selection);
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return __('filter::filter.search_term_name', ['term' => $selection]);
    }
}
