<?php

namespace Mtc\Filter\Filters;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Mtc\Filter\Contracts\IsFilter;
use Mtc\Shop\Brand;
use Mtc\Shop\Item;

class BrandFilter extends IsFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('brands', function (Builder $brand_query) use ($selection) {
            $brand_query->whereIn('brands.id', $selection)
                ->active();
        });
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        /** @var LengthAwarePaginator $results */
        $results =  Brand::query()
            ->active()
            ->whereHas('items', function ($item_query) use ($product_filtering) {
                return $product_filtering($item_query);
            })
            ->when(config('filter.get_filter_result_count'), function ($query) use ($product_filtering) {
                $query->orderByDesc('result_count')
                    ->addSelect([
                        'result_count' => Item::query()
                            ->selectRaw('count(*)')
                            ->join('items_brands', 'item_id', '=', 'items.id')
                            ->where(function ($item_query) use ($product_filtering) {
                                $product_filtering($item_query);
                            })
                            ->where(DB::raw('brands.id = `brand_id`')),
                    ]);
            })
            ->paginate($limit, ['*'], 'sidebar-page');

        $this->has_load_more = $results->lastPage() > 1 && $limit > 0;
        return collect($results->all());
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Brand::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return __('filter::filter.labels.brand');
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->name;
    }
}
