<?php

namespace Mtc\Filter\Contracts;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use RuntimeException;

abstract class FilterIndexContract extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'filter_index';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'slug',
        'name',
        'filter_type',
        'filter_id',
        'order',
    ];

    /**
     * Attributes visible when converting to json
     *
     * @var string[]
     */
    protected $visible = [
        'id',
        'slug',
        'name',
        'filter_type',
        'filter_id',
    ];

    protected static function boot()
    {
        parent::boot();

        self::addGlobalScope(fn($query) => $query->orderBy('order'));
    }

    /**
     * Relationship with filter model
     *
     * @return MorphTo
     */
    public function filter(): MorphTo
    {
        return $this->morphTo('filter');
    }

    /**
     * Index a model
     *
     * @param string $filter_type
     * @param Model $model
     * @param IsFilter $filter
     * @throws RuntimeException
     */
    public static function index(string $filter_type, Model $model, IsFilter $filter): void
    {
        $type = $filter_type;
        $id = $model->getAttribute($filter->getIdAttribute(true));
        $name = $model->getAttribute($filter->getNameAttribute(true));
        $slug = Str::slug($filter->modelSlug($model));

        if (empty(trim($id))) {
            return;
        }

        if (self::checkExisting($type, $id, $slug)) {
            $slug = self::uniqueSlug($type, $id, $slug);
        }

        self::query()
            ->updateOrCreate([
                'filter_type' => $type,
                'filter_id' => $id,
            ], [
                'slug' => $slug,
                'name' => $name
            ]);
    }

    /**
     * Check if slug is not already used
     *
     * @param string $type
     * @param $id
     * @param string $slug
     * @return bool
     */
    protected static function checkExisting(string $type, $id, string $slug): bool
    {
        $id = DB::getPdo()->quote($id);
        $type = DB::getPdo()->quote($type);
        return self::query()
            ->where('slug', $slug)
            ->where(DB::raw("NOT ( `filter_id` = $id and `filter_type` = $type)"))
            ->exists();
    }

    /**
     * Create a unique url slug for filter index element
     *
     * @param string $type
     * @param $id
     * @param string $slug
     * @return string
     * @throws RuntimeException
     */
    protected static function uniqueSlug(string $type, $id, string $slug): string
    {
        $slug_with_type = $slug . '-' . $type;
        if (self::checkExisting($type, $id, $slug_with_type) === false) {
            return $slug_with_type;
        }

        $i = 1;
        do {
            $iterated_slug = $slug . '-' . (++$i);

            if ($i > 100) {
                throw new RuntimeException('Unable to generate unique filter index slug');
            }
        } while (self::checkExisting($type, $id, $iterated_slug));

        return $iterated_slug;
    }
}
