<?php

namespace Mtc\Filter;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Mtc\Core\Facades\Media;
use Mtc\Filter\Contracts\FilterInstance;
use Mtc\Filter\Contracts\FilterSeoContract;
use Mtc\Filter\Contracts\SeoMatchingMechanism;
use Mtc\Shop\Brand;
use Mtc\Shop\Category;

class FilterSeo implements FilterSeoContract
{
    /**
     * Local variable that identifies whether match has been found
     *
     * @var bool
     */
    protected bool $matched = false;

    protected array $brands = [];
    protected array $categories = [];

    /**
     * Get Seo Data from Filter
     *
     * @param FilterInstance $filter
     * @return array
     */
    public function handle(FilterInstance $filter): array
    {
        $data =  $this->findSeoMatch($filter)->getData($filter);
        return array_merge($data, $this->getPageContent($filter, $data));
    }

    /**
     * Find the mechanism that should be used for getting SEO data
     *
     * @param FilterInstance $filter
     * @return SeoMatchingMechanism
     */
    protected function findSeoMatch(FilterInstance $filter): SeoMatchingMechanism
    {
        return collect(Config::get('filter.seo_data_matching', []))
            ->map(fn($matching_mechanism) => App::make($matching_mechanism))
            ->filter(function (SeoMatchingMechanism $mechanism) use ($filter) {
                return !$this->matched && ($this->matched = $mechanism->matchesRequest($filter));
            })
            ->first();
    }

    /**
     * Get on-page content for filter
     *
     * @param FilterInstance $filter
     * @param array $seo_data
     * @return array
     */
    protected function getPageContent(FilterInstance $filter, array $seo_data): array
    {
        return [
            'seo_title' => $this->getPageTitle($filter) ?? $seo_data['title'],
            'seo_description' => $this->getPageDescription($filter) ?? $seo_data['description'],
            'banner' => $this->getPageBanner($filter)
        ];
    }

    /**
     * Get title value for page
     *
     * @param FilterInstance $filter
     * @return string
     */
    protected function getPageTitle(FilterInstance $filter): string
    {
        $title = '';
        if (count($filter->getSelections('brand')) === 1) {
            $title = $this->loadBrand($filter->getSelections('brand')[0])->seo_title ?? '';
        }

        if (count($filter->getSelections('category')) > 0) {
            $categories = $filter->getSelections('category');
            $title = $this->loadCategory(array_pop($categories))->seo_title ?? '';
        }

        return $title;
    }

    /**
     * Get description value for page
     *
     * @param FilterInstance $filter
     * @return string
     */
    protected function getPageDescription(FilterInstance $filter): string
    {
        $description = '';
        if (count($filter->getSelections('brand')) === 1) {
            $description = $this->loadBrand($filter->getSelections('brand')[0])->seo_description ?? '';
        }

        if (count($filter->getSelections('category')) > 0) {
            $categories = $filter->getSelections('category');
            $description = $this->loadCategory(array_pop($categories))->seo_description ?? '';
        }

        return $description;
    }

    /**
     * Get banner value for page
     *
     * @param FilterInstance $filter
     * @return string
     */
    protected function getPageBanner(FilterInstance $filter): string
    {
        $banner = '';
        if (count($filter->getSelections('brand')) === 1) {
            $banner = $this->getBrandBanner($filter->getSelections('brand')[0]);
        }

        if (count($filter->getSelections('category')) > 0) {
            $categories = $filter->getSelections('category');
            $banner = $this->getCategoryBanner(array_pop($categories));
        }

        return $banner;
    }

    /**
     * Fetch Brand. store in object to avoid duplicate queries among title, description, image
     *
     * @param int $brand_id
     * @return Model|null
     */
    protected function loadBrand(int $brand_id)
    {
        if (empty($this->brands[$brand_id])) {
            $this->brands[$brand_id] = Brand::query()->find($brand_id);
        }
        return $this->brands[$brand_id];
    }

    /**
     * Fetch Category. store in object to avoid duplicate queries among title, description, image
     *
     * @param int $category_id
     * @return Model|null
     */
    protected function loadCategory(int $category_id)
    {
        if (empty($this->categories[$category_id])) {
            $this->categories[$category_id] = Category::query()->find($category_id);
        }
        return $this->categories[$category_id];
    }

    /**
     * Get banner url based on brand
     *
     * @param int $brand_id
     * @return string
     */
    protected function getBrandBanner(int $brand_id): string
    {
        $brand = $this->loadBrand($brand_id);
        $folders = Media::getFolders('brand_images');
        $size_path = $folders['normal']['path'];

        if (!empty($brand->image)) {
            return url("/$size_path/$brand->image");
        }

        $placeholder_images = \Mtc\Shop\PlaceholderImage::getPackedData();
        if (!empty($placeholder_images['brand']->value)) {
            return url("/$size_path/{$placeholder_images['brand']->value}");
        }

        return '';
    }

    /**
     * Get banner url based on category
     *
     * @param int $category_id
     * @return string
     */
    protected function getCategoryBanner(int $category_id): string
    {
        $category = $this->loadCategory($category_id);
        $folders = Media::getFolders('brand_images');
        $size_path = $folders['normal']['path'];

        if (!empty($category->image)) {
            return url("/$size_path/$category->image");
        }

        $placeholder_images = \Mtc\Shop\PlaceholderImage::getPackedData();
        if (!empty($placeholder_images['category']->value)) {
            return url("/$size_path/{$placeholder_images['category']->value}");
        }

        return '';
    }
}
