<?php

namespace Mtc\Foundation;

use Illuminate\Database\Eloquent\Model;

/**
 * Class Address
 *
 * Foundation address that can implements base functionality across different address usages
 *
 * @package Mtc\Foundation
 */
class Address extends Model
{
    /**
     * Address type constants
     */
    const TYPE_SHIPPING = 'shipping';
    const TYPE_BILLING = 'billing';
    const TYPE_COLLECTION = 'collection';

    /**
     * Columns that cannot be mass assigned
     *
     * @var array
     */
    protected $guarded = [
        'id'
    ];

    /**
     * Address blueprint - empty address array
     *
     * @var array
     */
    public static $blueprint = [
        'type' => '',
        'first_name' => '',
        'last_name' => '',
        'address1' => '',
        'address2' => '',
        'city' => '',
        'county' => '',
        'state' => '',
        'postcode' => '',
        'country' => '',
    ];

    /**
     * Fields that identify address (excluding name)
     *
     * @var array
     */
    public static $address_fields = [
        'address1',
        'address2',
        'city',
        'county',
        'state',
        'postcode',
        'country',
    ];

    /**
     * Relationship to the country that has this code
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function ownerCountry()
    {
        return $this->belongsTo(Country::class, 'country', 'code');
    }

    /**
     * Get all filled address fields
     *
     * @return \Illuminate\Support\Collection
     */
    public function getFullAddressAttribute()
    {
        return collect($this->only(self::$address_fields))->filter()
            ->reject(function ($value) {
                return $value === 'NA';
            });
    }

    /**
     * Get Address fields separated with comma
     *
     * @return string
     */
    public function getFormattedAddressAttribute()
    {
        return $this->getFullAddressAttribute()->implode(', ');
    }

    /**
     * Quick access to customers full name
     *
     * @return string
     */
    public function getCustomerNameAttribute()
    {
        return $this->first_name . ' ' . $this->last_name;
    }
}
