<?php

namespace Mtc\FreedomPay;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;
use Mtc\Core\Models\Country;
use Illuminate\Support\Facades\Hash;

/**
 * Class FreedomPay
 *
 * @package Mtc\Plugins\FreedomPay\Classes
 */
class FreedomPay implements PaymentGateway
{
    protected $client;

    /**
     * @param Client $client
     */
    public function __construct(Client $client)
    {
        $this->client = $client;
        ini_set('precision', 17);
        ini_set('serialize_precision', -1);
    }

    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     *
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        // check if payment is actually needed
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('freedompay.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $details = $this->createTransaction($this->mapToTransaction($invoice, $invoice->getPayable()));
        if (empty($details)) {
            return new PaymentForm('', '');
        }

        $template = template('freedompay/freedompay_payment_form.twig', [
            'action' => $details['checkout_url'],
            'transaction_id' => $details['transaction_id'],
            'button_label' => __('freedompay::freedompay.pay_now'),
            'production' => config('freedompay.use_production_endpoint')
        ]);

        return new PaymentForm($template, 'template', [
            'name' => __('freedompay::freedompay.payment_option_name'),
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     *
     * @return array
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        // Payment flow is handled in FreedomPayController
        return [];
    }

    /**
     * Get the endpoint URL based on environment
     *
     * @param $live_mode
     *
     * @return string
     */
    protected function endpoint(string $path, $live_mode): string
    {
        return $live_mode
            ? 'https://payments.freedompay.com/CheckoutService/CheckoutService.svc' . $path
            : 'https://payments.uat.freedompay.com/CheckoutService/CheckoutService.svc' . $path;
    }

    /**
     * @param InvoiceRepositoryContract $invoice
     * @param $payable
     *
     * @return array
     */
    protected function createTransaction($transaction): array
    {
        $response = $this->makeRequest('/CreateTransaction', $transaction);

        // validate response
        if (empty($response['error']) && !empty($response['CheckoutUrl']) && !empty($response['TransactionId'])) {
            return [
                'transaction_id' => $response['TransactionId'],
                'checkout_url' => $response['CheckoutUrl'],
            ];
        }

        return [];
    }

    /**
     * @param string $transaction_id
     *
     * @return array
     */
    public function getTransaction($transaction_id = ''): array
    {
        if (empty($transaction_id)) {
            return [];
        }

        return $this->makeRequest('/GetTransaction', $transaction_id);
    }

    /**
     * @param string $transaction_id
     *
     * @return array
     */
    public function cancelTransaction($transaction_id = '')
    {
        if (empty($transaction_id)) {
            return [];
        }

        return $this->makeRequest('/CancelTransaction', $transaction_id);
    }

    /**
     * @param InvoiceRepositoryContract $invoice
     * @param $payable
     *
     * @return string[]
     */
    protected function mapToTransaction(InvoiceRepositoryContract $invoice, $payable): array
    {
        $billing = $shipping = $invoice->getModel()->recipient;
        $billing_country_iso_3116_1 = Country::query()
            ->select('iso_code_3166_1_numeric')
            ->where('code', $billing->country)
            ->first()
            ->iso_code_3166_1_numeric;

        $shipping_country_iso_3116_1 = Country::query()
            ->select('iso_code_3166_1_numeric')
            ->where('code', $shipping->country)
            ->first()
            ->iso_code_3166_1_numeric;

        return [
            'StoreId' => config('freedompay.freedom_pay_store_id'),
            'TerminalId' => config('freedompay.freedom_pay_store_terminal_id'),
            'TransactionTotal' => sprintf("%.2f", $invoice->getOutstandingAmount()),
            'TaxTotal' => sprintf("%.2f", $invoice->getModel()->vat_value),
            'CustomerCode' => crc32($invoice->getEmail()),
            'InvoiceNumber' => $invoice->getId(),
            'MerchantReferenceCode' => $invoice->getReference(),
            'CaptureMode' => true,
            'AddressRequired' => false,
            'AllowInternationalAddresses' => true,
            'CustomerEmail' => $invoice->getEmail(),
            'BillingAddress' => [
                'Name' => $billing->first_name . ' ' . $billing->last_name,
                'Street1' => $billing->address1,
                'Street2' => $billing->address2 ?? '',
                'City' => $billing->city ?? '',
                'State' => $billing->state ?? '',
                'PostalCode' => $billing->postcode ?? '',
                'CountryCode' => $billing_country_iso_3116_1 ?? '',
            ],
            'Fields' => [
                0 => [
                    'Key' => 'BillingAddress1',
                    'Value' => $billing->address1,
                ],
                1 => [
                    'Key' => 'BillingCity',
                    'Value' => $billing->city,
                ],
                2 => [
                    'Key' => 'BillingCountryCode',
                    'Value' => $billing_country_iso_3116_1,
                ],
                3 => [
                    'Key' => 'BillingFirstName',
                    'Value' => $billing->first_name,
                ],
                4 => [
                    'Key' => 'BillingLastName',
                    'Value' => $billing->last_name,
                ],
                5 => [
                    'Key' => 'BillingPostalCode',
                    'Value' => $billing->postcode,
                ],
                6 => [
                    'Key' => 'MobilePhone',
                    'Value' => $invoice->getModel()->contact_number,
                ],
            ],
            'ShipToAddress' => [
                'Name' => $shipping->first_name . ' ' . $shipping->last_name,
                'Street1' => $shipping->address1,
                'Street2' => $shipping->address2 ?? '',
                'City' => $shipping->city ?? '',
                'State' => $shipping->state ?? '',
                'PostalCode' => $shipping->postcode ?? '',
                'CountryCode' => $shipping_country_iso_3116_1 ?? '',
            ],
            'ShowAddress' => false,
            'PurchaseItems' => $this->mapPurchaseItems($invoice->getModel()->items),
            'InvoiceItems' => $this->mapInvoiceItems($invoice->getModel()->items, $payable->surcharges),
            'LevelThreeItems' => $this->mapLevelItems($invoice->getModel()->items)
        ];
    }

    /**
     * @param $items
     *
     * @return array
     */

    public function mapLevelItems($items): array
    {
        $level_three_items = [];

        if (!empty($items)) {
            $i = 1;
            foreach ($items as $item) {
                $level_three_item = [
                    'AllocCode' => null,
                    'CommodityCode' => null,
                    'Custom1' => null,
                    'Custom2' => null,
                    'Custom3' => null,
                    'Custom4' => null,
                    'Custom5' => null,
                    'Custom6' => null,
                    'Custom7' => null,
                    'Custom8' => null,
                    'Custom9' => null,
                    'CustomFormatId' => null,
                    'CustomerAssetId' => null,
                    'DiscountAmount' => null,
                    'DiscountFlag' => false,
                    'EidIndicator' => null,
                    'FreightAmount' => 0,
                    'Id' => null,
                    'OrigTotalAmount' => $item->line_total->withTax(),
                    'OrigUnitPrice' => $item->value->withoutTax(),
                    'PayAlloc' => 0,
                    'ProductCode' => $item->id,
                    'ProductDescription' => $item->name . ' - ' . json_encode($item->details),
                    'ProductMake' => null,
                    'ProductModel' => null,
                    'ProductName' => Str::limit($item->name, 32),
                    'ProductPartNumber' => null,
                    'ProductSKU' => $item->getPurchasableSkuAttribute(),
                    'ProductSerial1' => null,
                    'ProductSerial2' => null,
                    'ProductSerial3' => null,
                    'ProductUPC' => null,
                    'ProductYear' => 0,
                    'PromoCode' => null,
                    'Quantity' => 1,
                    'SaleCode' => "S",
                    'Tag' => null,
                    'TaxAmount' => $item->value->tax(),
                    'TaxIncludedFlag' => false,
                    'TotalAmount' => $item->value->withTax(),
                    'UnitOfMeasure' => null,
                    'UnitPrice' => $item->value->withoutTax()
                ];
                $i++;
                $level_three_items[] = $level_three_item;
            }
        }
        return $level_three_items;
    }

    /**
     * @param $items
     *
     * @return array
     */
    public function mapPurchaseItems($items): array
    {
        $purchase_items = [];

        if ($items !== null) {
            $i = 1;
            foreach ($items as $item) {
                $purchase_item = [
                    'Description' => $item->name,
                    'DisplayOrder' => $i,
                    'Price' => number_format($item->value->withTax(), 2, '.', ''),
                    'Quantity' => $item->quantity
                ];

                $purchase_items[] = $purchase_item;
                $i++;
            }
        }

        return $purchase_items;
    }

    /**
     * @param $items
     * @param $surcharges
     *
     * @return array
     */
    public function mapInvoiceItems($items, $surcharges): array
    {
        $invoice_items = [];

        if (!empty($items)) {
            $i = 1;
            foreach ($items as $item) {
                $invoice_item = [
                    'DisplayOrder' => $i,
                    'FreewayItemType' => 'None',
                    'IsVisible' => true,
                    'Label' => $item->name,
                    'Value' => number_format($item->line_total->withTax(), 2, '.', ''),
                ];

                $invoice_items[] = $invoice_item;
                $i++;
            }

            if ($surcharges !== null) {
                foreach ($surcharges as $surcharge) {
                    if ((bool)$surcharge['is_negative'] === false) {
                        $invoice_item = [
                            'DisplayOrder' => $i,
                            'FreewayItemType' => $surcharge->surcharge_type === 'delivery' ? 'Shipping' : 'None',
                            'IsVisible' => true,
                            'Label' => $surcharge->name,
                            'Value' => number_format($surcharge->surcharge_amount->raw(), 2, '.', ''),

                        ];

                        $invoice_items[] = $invoice_item;
                    }
                }
            }
        }

        return $invoice_items;
    }

    /**
     * @param string $transaction_type
     * @param $payload
     *
     * @return array
     */
    protected function makeRequest(string $path, $payload)
    {
        try {
            $response = $this->client->post($this->endpoint($path, config('freedompay.use_production_endpoint')), [
                'headers' => [
                    'Content-Type: application/json',
                    'Accept: application/json',
                    'Cache-Control: no-cache',
                ],
                'json' => $payload
            ]);


            return json_decode((string)$response->getBody(), true);
        } catch (ClientException  $exception) {

            Log::error('Failed to register transaction', [
                'code' => $exception->getCode(),
                'message' => $exception->getMessage(),
            ]);
            return [];
        } catch (\Exception  $exception) {
            Log::error('Failed to register transaction', [
                'code' => $exception->getCode(),
                'message' => $exception->getMessage(),
            ]);
            return [];
        }
    }
}