<?php

namespace Mtc\FreedomPay\Http\Controllers;

use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\URL;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\FreedomPay\FreedomPay;

class FreedomPayController extends Controller
{
    /**
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     *
     * @return RedirectResponse|string
     */
    public function index(Request $request, InvoiceRepositoryContract $invoice, FreedomPay $freedomPay)
    {
        // redirect to the failed payment page if missing mandatory params
        if (empty($request->input('transid')) || empty($request->input('status'))) {
            return $this->redirectToFailed($request->input('status'));
        }

        // check if request is to cancel the transaction
        if ($request->input('status') === 'C') {
            $freedomPay->cancelTransaction($request->input('transid'));
            return $this->redirectToFailed($request->input('status'));
        }

        $response = $freedomPay->getTransaction($request->input('transid'));

        // process the payment
        if (empty($response['InvoiceNumber'])) {
            throw new \RuntimeException("FreedomPay transaction failed. The invoice number is missing!");
        }

        // load up the order info
        $invoice->load($response['InvoiceNumber']);

        if ($invoice->isPaid()) {
            return $this->redirectToSuccess($invoice);
        }

        if ($this->isSuccessfulResponse($response)) {
            $invoice->savePayment([
                'provider' => 'freedompay',
                'amount' => $response['OriginalRequest']['TransactionTotal'],
                'reference' => $request->input('transid'),
                'details' => $response,
                'confirmed_at' => now()
            ]);
            return $this->redirectToSuccess($invoice);
        }

        return $this->redirectToFailed($request->input('status'));
    }

    /**
     * @param Request $request
     *
     * @return array|string[]
     */
    public function checkTransaction(Request $request, FreedomPay $freedomPay): array
    {
        if (empty($request->input('transid'))) {
            return [
                'error' => 'Missing Transaction ID!'
            ];
        }

        return $freedomPay->getTransaction($request->input('transid'));
    }

    /**
     * Render the failure page.
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     *
     * @return string
     */
    public function failed(Request $request, InvoiceRepositoryContract $invoice): string
    {
        return App::make(config('checkout.payment_response_handler'))->failure($invoice);
    }

    /**
     * Return redirect to failure page
     *
     * @param $status
     * @return RedirectResponse
     */
    private function redirectToFailed($status)
    {
        return redirect()->to(
            URL::signedRoute(
                'freedompay.failed_transaction',
                ['status_code' => $status],
                null,
                config('app.env') === 'production'
            )
        );
    }

    /**
     * Return redirect to success page
     *
     * @param InvoiceRepositoryContract $invoice
     * @return RedirectResponse
     */
    private function redirectToSuccess(InvoiceRepositoryContract $invoice)
    {
        return redirect()->to(
            URL::signedRoute(
                'successful_payment',
                ['id' => $invoice->getId()],
                null,
                config('app.env') === 'production'
            )
        );
    }

    /**
     * Check if response from FreedomPay was successful for transaction
     *
     * @param array $response
     * @return bool
     */
    private function isSuccessfulResponse(array $response)
    {
        return isset($response['AuthResponse'], $response['CaptureResponse'])
            && $response['AuthResponse']['AuthorizationDecision'] === 'ACCEPT'
            && $response['CaptureResponse']['AuthorizationDecision'] === 'ACCEPT';
    }
}
