<?php

namespace Mtc\GpAddresses\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Http\Client\Pool;
use Illuminate\Http\Client\ConnectionException;
use Mtc\GpAddresses\Models\GpAddress;
use Mtc\GpAddresses\Models\NhsOrganisation;

use Illuminate\Support\Facades\Log;

class FetchGpAddresses extends Command
{
    protected $signature = 'gp:fetch-addresses';

    protected $description = 'Fetch NHS organisations from the API.';

    public function handle(): void
    {
        Log::info("Starting fetch of NHS Addresses");

        $orgLinks = NhsOrganisation::pluck('org_link');

        $orgLinks->chunk(50)->each(function ($chunk) {

            $responses = Http::pool(function (Pool $pool) use ($chunk) {
                $requests = [];

                foreach ($chunk as $link) {
                    $requests[] = $pool->get($link);
                }

                return $requests;
            });

            foreach ($responses as $response) {
                if ($response->successful()) {
                    $data = $response->json();

                    $orgId = $data['Organisation']['OrgId']['extension'] ?? null;

                    if (!$orgId) {
                        Log::warning("Organisation missing OrgId extension, skipping");
                        continue;
                    }

                    $org = NhsOrganisation::where('org_id', $orgId)->value('id');

                    if (!$org) {
                        Log::warning("Organisation not found for OrgId extension {$orgId}, skipping");
                        continue;
                    }

                    $location = $data['Organisation']['GeoLoc']['Location'] ?? null;

                    if ($location) {
                        $uprn = $location['UPRN'] ?? null;

                        if (!$uprn) {
                            Log::info("Missing a UPRN.");
                        }

                        $exists = GpAddress::where('nhs_organisation_id', $org)
                            ->where('address_line_1', $location['AddrLn1'] ?? null)
                            ->where('postcode', $location['PostCode'] ?? null)
                            ->exists();

                        if (!$exists) {
                            GpAddress::create([
                                'uprn'                 => $uprn,
                                'nhs_organisation_id'  => $org,
                                'address_line_1'       => $location['AddrLn1'] ?? null,
                                'address_line_2'       => $location['AddrLn2'] ?? null,
                                'town'                 => $location['Town'] ?? null,
                                'county'               => $location['County'] ?? null,
                                'postcode'             => $location['PostCode'] ?? null,
                                'country'              => $location['Country'] ?? 'England',
                            ]);
                        }
                    } else {
                        Log::warning("Organisation {$orgId} is missing location info, has it been deleted?.");
                    }
                } elseif ($response->status() == 429) {
                    Log::warning('Rate Limited, waiting 5 seconds...');
                    sleep(5);
                }
            }
        });

        Log::info("Finished fetch of NHS Addresses.");
    }
}
