<?php

namespace Mtc\GpAddresses\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Http\Client\ConnectionException;
use Mtc\GpAddresses\Models\GpAddress;
use Mtc\GpAddresses\Models\NhsOrganisation;

use Illuminate\Support\Facades\Log;

class FetchNhsOrganisations extends Command
{
    protected $signature = 'gp:fetch-organisations';

    protected $description = 'Fetch NHS organisations from the API.';

    public function handle(): void
    {
        Log::info("Starting NHS organisations fetch.");

        $url = config('gpaddresses.url');

        while ($url) {
            try {
                // NHS API is slow, need to be gentle with it
                $response = Http::retry(3, 10000)
                    ->timeout(60)
                    ->connectTimeout(10)
                    ->get($url)
                    ->throw();

                $batch = collect($response->json()['Organisations'] ?? []);

                foreach ($batch as $org) {
                    if (empty($org['OrgId']) || empty($org['Name'])) continue;
                    
                    NhsOrganisation::updateOrCreate(
                        ['org_id' => $org['OrgId']],
                        [
                            'name' => $org['Name'],
                            'status' => $org['Status'] ?? null,
                            'org_link' => $org['OrgLink'] ?? null,
                        ]
                    );
                }

                Log::info("Records have finished being inserted/updated at {$url}");
                $url = $response->header('next-page') ?: null;
                if($url) {
                    Log::info("Moving on to {$url}");
                }
            } catch (\Illuminate\Http\Client\RequestException $e) {
                if ($e->response && $e->response->status() == 429) {
                    Log::info('Rate limit hit, sleeping 5 seconds...');
                    sleep(5);
                    continue;
                }

                Log::error("Request failed: " . $e->getMessage());
                break;
            }
        }

        Log::info("Finished fetching organisations. Now attempting to store organisations...");

        $oldCount = NhsOrganisation::all()->count();

        $newCount = NhsOrganisation::all()->count();

        Log::info("Finished storing organisations.");

        if ($newCount > 0) {
            $added = $newCount - $oldCount;
            Log::info("Added {$added} new organisations. Total now: {$newCount}");
        } else {
            Log::info("No new records were added.");
        }
    }
}
