<?php

namespace Mtc\GpAddresses\Jobs;

use Illuminate\Console\Command;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\GpAddresses\Models\GpAddress;
use Mtc\GpAddresses\Models\NhsOrganisation;

class DeleteAddresses implements ShouldQueue {
    use Queueable;

    public function handle(): void
    {
        Log::info("Starting to delete GP addresses.");

        $url = config('gpaddresses.deleted');

        $response = Http::get($url);

        if (!$response->successful()) {
            Log::error("Failed to fetch deleted organisations: HTTP {$response->status()}");
            return;
        }

        // Handle downloaded file or normal JSON
        $raw = $response->body();
        $data = json_decode($raw, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            Log::error("Failed to decode JSON from response: " . json_last_error_msg());
            return;
        }

        $deletedOrgs = collect($data['DeletedOrganisations'] ?? []);

        if ($deletedOrgs->isEmpty()) {
            Log::info("No deleted organisations found.");
            return;
        }

        // Extract OrgIDs
        $orgIds = $deletedOrgs->pluck('OrgID')->filter()->values();

        Log::info("Found " . $orgIds->count() . " organisations marked as deleted.");

        // Map to local organisation IDs
        $localOrgIds = NhsOrganisation::whereIn('org_id', $orgIds)->pluck('id');

        if ($localOrgIds->isEmpty()) {
            Log::info("None of the deleted organisations exist locally.");
            return;
        }

        // Delete associated GP addresses
        $deletedCount = GpAddress::whereIn('nhs_organisation_id', $localOrgIds)->delete();

        Log::info("Deleted {$deletedCount} GP addresses linked to deleted organisations.");

        Log::info("Finished deleting GP addresses.");
    }
}
