<?php

namespace Mtc\GpAddresses\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\GpAddresses\Models\GpAddress;
use Mtc\GpAddresses\Models\NhsOrganisation;

class FetchAddresses implements ShouldQueue
{
    use Queueable;

    public function handle(): void
    {
        Log::info("Fetching addresses...");

        $links = NhsOrganisation::pluck('org_link');
        $batchSize = 50; // 50 requests per batch
        $sleepSeconds = 10; // 50 every 10s = 300/min

        $prev = GpAddress::count();

        foreach ($links->chunk($batchSize) as $chunk) {
            $responses = Http::pool(
                fn($pool) =>
                $chunk->map(fn($link) => $pool->get($link))->all()
            );

            $addresses = [];

            foreach ($responses as $response) {
                if ($response->successful()) {
                    $data = $response->json();
                    $orgId = $data['Organisation']['OrgId']['extension'] ?? null;
                    $org = NhsOrganisation::where('org_id', $orgId)->value('id');
                    $location = $data['Organisation']['GeoLoc']['Location'] ?? null;

                    if ($orgId && $location) {
                        $addresses[] = [
                            'nhs_organisation_id' => $org,
                            'address_line_1'      => $location['AddrLn1'] ?? null,
                            'postcode'            => $location['PostCode'] ?? null,
                            'uprn'                => $location['UPRN'] ?? null,
                            'address_line_2'      => $location['AddrLn2'] ?? null,
                            'town'                => $location['Town'] ?? null,
                            'county'              => $location['County'] ?? null,
                            'country'             => $location['Country'] ?? 'England',
                            'updated_at'          => now(),
                            'created_at'          => now(),
                        ];
                    }
                } elseif ($response->tooManyRequests()) {
                    Log::warning("Rate limit hit, skipping this batch. Will continue with next batch.");
                    break; // skip the rest of this batch
                }
            }

            if (!empty($addresses)) {
                GpAddress::upsert(
                    $addresses,
                    ['nhs_organisation_id', 'address_line_1', 'postcode'],
                    ['uprn', 'address_line_2', 'town', 'county', 'country', 'updated_at']
                );
            }

            sleep($sleepSeconds); // throttle to respect 300/min
        }

        $total = GpAddress::count() - $prev;

        Log::info("Inserted {$total} new addresses.");
    }
}
