<?php

namespace Mtc\GpAddresses\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\GpAddresses\Models\NhsOrganisation;

class FetchOrganisations implements ShouldQueue
{
    use Queueable;

    public function handle(): void
    {
        Log::info("Fetching Organisations...");

        $url = config('gpaddresses.fetch');

        while ($url) {
            try {
                $response = Http::retry(3, 10000)
                    ->get($url)
                    ->throw();

                $orgs = [];

                if ($response->successful()) {
                    $json = $response->json()['Organisations'] ?? [];

                    foreach ($json as $data) {
                        $org = [
                            'org_id' => $data['OrgId'] ?? null,
                            'name' => $data['Name'] ?? null,
                            'status' => $data['Status'] ?? null,
                            'org_link' => $data['OrgLink'] ?? null
                        ];

                        $hasNull = in_array(null, $org, true);

                        if ($hasNull) {
                            Log::warning("Response contained a NULL value. Skipping that response.");
                            continue;
                        }

                        $orgs[] = $org;
                    }

                    $url = $response->header('next-page') ?: null;

                    if ($url) {
                        Log::info("Moving on to {$url}");
                    }
                }

                NhsOrganisation::upsert(
                    $orgs,
                    ['org_id'],          // unique key
                    ['name', 'status', 'org_link'] // columns to update
                );
            } catch (\Illuminate\Http\Client\RequestException $e) {
                if ($e->response && $e->response->status() == 429) {
                    Log::info('Rate limit hit, sleeping 5 seconds...');
                    sleep(5);
                    continue;
                }

                Log::error("Request failed: " . $e->getMessage());
                break;
            }
        }
    }
}
