<?php

namespace Mtc\GpAddresses\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Http\Client\Pool;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Mtc\GpAddresses\Models\GpAddress;

class UpdateAddresses implements ShouldQueue
{
    use Queueable;

    public function handle(): void
    {
        $jobName = __METHOD__;

        $jobRun = DB::table('gpaddresses_job_runs')->where('job_name', $jobName)->first();

        if ($jobRun) {
            // If already ran today, skip
            if ($jobRun->last_run_at === now()->toDateString()) {
                Log::info("{$jobName} already ran today. Skipping.");
                return;
            }

            $lastRunDate = $jobRun->last_run_at;
        } else {
            // First ever run
            DB::table('gpaddresses_job_runs')->insert([
                'job_name'    => $jobName,
                'last_run_at' => now()->toDateString(),
                'created_at'  => now(),
                'updated_at'  => now(),
            ]);

            $lastRunDate = now()->toDateString();
        }

        $url = config('gpaddresses.update') . $lastRunDate;

        $response = Http::get($url);
        if (!$response->successful()) {
            Log::error("Failed to fetch update list from {$url}");
            return;
        }

        $data = $response->json();
        $collection = collect($data['Organisations'] ?? [])->pluck('OrgLink');

        $collection->chunk(50)->each(function ($chunk) {
            $responses = Http::pool(function (Pool $pool) use ($chunk) {
                return $chunk->map(fn($link) => $pool->get($link))->all();
            });

            $addresses = [];

            foreach ($responses as $response) {
                if ($response instanceof \Illuminate\Http\Client\Response) {
                    if ($response->successful()) {
                        $data = $response->json();
                        $orgId = $data['Organisation']['OrgId']['extension'] ?? null;

                        if (!$orgId) {
                            Log::warning("Organisation missing OrgId extension, skipping");
                            continue;
                        }

                        $org = NhsOrganisation::where('org_id', $orgId)->value('id');
                        if (!$org) continue;

                        Log::info("Updating organisation {$orgId}.");

                        $location = $data['Organisation']['GeoLoc']['Location'] ?? null;
                        if (!$location) {
                            Log::warning("Organisation {$orgId} missing location info, skipping.");
                            continue;
                        }

                        $addresses[] = [
                            'nhs_organisation_id' => $org,
                            'address_line_1'      => $location['AddrLn1'] ?? null,
                            'postcode'            => $location['PostCode'] ?? null,
                            'uprn'                => $location['UPRN'] ?? null,
                            'address_line_2'      => $location['AddrLn2'] ?? null,
                            'town'                => $location['Town'] ?? null,
                            'county'              => $location['County'] ?? null,
                            'country'             => $location['Country'] ?? 'England',
                            'updated_at'          => now(),
                            'created_at'          => now(),
                        ];
                    } elseif ($response->status() === 429) {
                        Log::warning('Rate limited, waiting 5 seconds...');
                        sleep(5);
                    } else {
                        Log::warning("HTTP request failed with status {$response->status()}");
                    }
                } else {
                    Log::error("Connection failed: " . $response->getMessage());
                }
            }

            if (!empty($addresses)) {
                GpAddress::upsert(
                    $addresses,
                    ['nhs_organisation_id', 'address_line_1', 'postcode'],
                    ['uprn', 'address_line_2', 'town', 'county', 'country', 'updated_at']
                );
            }
        });

        DB::table('gpaddresses_job_runs')
            ->where('job_name', $jobName)
            ->update([
                'last_run_at' => now()->toDateString(),
                'updated_at'  => now(),
            ]);

        Log::info("{$jobName} finished and last_run_at updated.");
    }
}
