<?php

namespace Mtc\GpAddresses\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Mtc\GpAddresses\Models\GpAddress;

class FetchAddresses implements ShouldQueue
{
    use Queueable;

    public function __construct(public $links) {}

    public function handle(): void
    {
        $addresses = [];

        foreach ($this->links as $link) {

            $attempts = 0;

            do {
                $response = Http::get($link);

                $attempts++;

                Log::info("Fetching $link, attempt #$attempts, status {$response->status()}.");

                if ($response->ok()) {
                    $orgId = $response->json('Organisation.OrgId.extension') ?? null;
                    $org = NhsOrganisation::where('org_id', $orgId)->value('id');
                    $location = $response->json('Organisation.GeoLoc.Location') ?? null;

                    if ($orgId && $location) {
                        $addresses[] = [
                            'nhs_organisation_id' => $org,
                            'address_line_1'      => $location['AddrLn1'] ?? null,
                            'postcode'            => $location['PostCode'] ?? null,
                            'uprn'                => $location['UPRN'] ?? null,
                            'address_line_2'      => $location['AddrLn2'] ?? null,
                            'town'                => $location['Town'] ?? null,
                            'county'              => $location['County'] ?? null,
                            'country'             => $location['Country'] ?? 'England',
                            'updated_at'          => now(),
                            'created_at'          => now(),
                        ];
                    }
                    break;
                }

                sleep(2);
            } while ($response->status() === 429 && $attempts < 5);
        }

        if (!empty($addresses)) {
            GpAddress::upsert(
                $addresses,
                ['nhs_organisation_id', 'address_line_1', 'postcode'],
                ['uprn', 'address_line_2', 'town', 'county', 'country', 'updated_at']
            );
        }
    }
}
