<?php

namespace Mtc\GpAddresses\Services;

use Illuminate\Support\Facades\Http;
use Mtc\GpAddresses\Models\GpAddress;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Illuminate\Support\Facades\Log;
use Mtc\GpAddresses\Jobs\FetchAddresses;
use Illuminate\Support\Facades\DB;

class GpAddressesService
{
    /**
     * Fetch and store organisations in the database.
     *
     * @param $dispatchAddresses bool Option to dispatch jobs that can pull addresses in the background after we pull all the organisations.
     * @return void
     * */
    public function fetchAndStore(bool $dispatchAddresses = true): void
    {
        $url = config('gpaddresses.fetch');

        $attempts = 0;

        $orgs = [];

        while ($url) {
            $attempts = 0;

            // Handle url
            do {
                $response = Http::get($url);
                $attempts++;

                Log::info("Fetching data from $url, attempt #$attempts, status {$response->status()}");

                // Handle data
                if ($response->ok()) {
                    $json = $response->json('Organisations') ?? [];
                    foreach ($json as $data) {
                        $org = [
                            'org_id' => $data['OrgId'] ?? null,
                            'name' => $data['Name'] ?? null,
                            'status' => $data['Status'] ?? null,
                            'org_link' => $data['OrgLink'] ?? null
                        ];

                        if (in_array(null, $org, true)) {
                            Log::warning("Organisation missing values. Skipping.");
                            continue;
                        }

                        $orgs[] = $org;
                    }

                    // Move to next page
                    $url = $response->header('next-page') ?: null;
                    break; // success, exit retry loop
                }

                sleep(2); // backoff on 429

            } while ($response->status() === 429 && $attempts < 5);

            // If we hit max attempts on this page, stop paging
            if ($attempts >= 5 && $response->status() === 429) {
                Log::warning("Max retries reached for $url. Stopping.");
                break;
            }
        }

        NhsOrganisation::upsert(
            $orgs,
            ['org_id'],          // unique key
            ['name', 'status', 'org_link'] // columns to update
        );

        // If true, dispatch a bunch of jobs to run in the background
        if ($dispatchAddresses) {
            $links = collect(array_column($orgs, 'org_link'));

            $batchSize = 100;

            $totalChunks = ceil($links->count() / $batchSize);

            $links->chunk($batchSize)->each(function ($chunk, $index) use ($totalChunks) {
                // Dispatch the jobs to job queue
                // Means app can get on with other tasks whilst we are still adding addresses
                Log::info("Dispatching batch " . ($index + 1) . " of $totalChunks, chunk size: " . count($chunk));
                dispatch(new FetchAddresses($chunk));
            });
        }
    }

    public function sync(): void
    {
        $jobName = "SyncAddresses";

        // Attempt to get when the job last ran
        $date = DB::table('gpaddresses_job_runs')
            ->where('job_name', $jobName)
            ->first();

        if (! $date) {
            $id = DB::table('gpaddresses_job_runs')->insertGetId([
                'job_name' => $jobName,
                'last_run_at' => now(),
            ]);
            $date = DB::table('gpaddresses_job_runs')->where('id', $id)->first();
        }

        $date = $date->last_run_at;

        // Set the parameter for the url
        $url = config('gpaddresses.sync') . $date;

        $links = [];

        $attempts = 0;

        // Fetch all the links for addresses that need updated
        do {
            $response = Http::get($url);

            $attempts++;

            Log::info("Fetching Links from $url, attempt #$attempts, status {$response->status()}");

            if ($response->ok()) {
                $links = $response->json('Organisations');
                break;
            }
        } while ($response->status() === 429 && $attempts < 5);

        if ($attempts >= 5) {
            Log::info("Failed to fetch data.");
            return;
        }

        $count = count($links);

        Log::info("Fetched a total of {$count} links.");


        $addresses = [];

        // Attempt to get updated data for Organisations we have
        foreach ($links as $item) {
            $link = $item['OrgLink'] ?? null;
            if (! $link) continue;
            // Reset attempts when trying to get each link
            $attempts = 0;

            do {
                $response = Http::get($link);

                $attempts++;

                if ($response->ok()) {

                    $orgId = $response->json('Organisation.OrgId.extension') ?? null;
                    $org = NhsOrganisation::firstWhere('org_id', $orgId);
                    $location = $response->json('Organisation.GeoLoc.Location') ?? null;

                    // If the API has an ID
                    // And we have a matching organisation
                    // And we successfully found the location from the API
                    if ($orgId && $org && $location) {
                        Log::info("Fetching data for {$org->name} from $link, attempt #$attempts, status {$response->status()}");

                        $addresses[] = [
                            'nhs_organisation_id' => $org->id,
                            'address_line_1'      => $location['AddrLn1'] ?? null,
                            'postcode'            => $location['PostCode'] ?? null,
                            'uprn'                => $location['UPRN'] ?? null,
                            'address_line_2'      => $location['AddrLn2'] ?? null,
                            'town'                => $location['Town'] ?? null,
                            'county'              => $location['County'] ?? null,
                            'country'             => $location['Country'] ?? 'England',
                            'updated_at'          => now(),
                            'created_at'          => now(),
                        ];

                        break;
                    } else {
                        Log::warning("Skipping {$orgId}, no address is stored for it.");
                        continue;
                    }
                }
            } while ($response->status() === 429 && $attempts < 5);
        }

        if (!empty($addresses)) {
            GpAddress::upsert(
                $addresses,
                ['nhs_organisation_id', 'address_line_1', 'postcode'],
                ['uprn', 'address_line_2', 'town', 'county', 'country', 'updated_at']
            );
        }

        // Update table to today
        DB::table('gpaddresses_job_runs')
            ->where('job_name', $jobName)
            ->update([
                'last_run_at' => now()->toDateString(),
                'updated_at'  => now(),
            ]);

        Log::info("Finished updating the GP Addresses.");
    }

    public function delete(): void
    {
        Log::info("Starting to delete GP addresses.");

        $url = config('gpaddresses.deleted');

        $response = Http::get($url);

        if (!$response->successful()) {
            Log::error("Failed to fetch deleted organisations: HTTP {$response->status()}");
            return;
        }

        // Handle downloaded file or normal JSON
        $raw = $response->body();
        $data = json_decode($raw, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            Log::error("Failed to decode JSON from response: " . json_last_error_msg());
            return;
        }

        $deletedOrgs = collect($data['DeletedOrganisations'] ?? []);

        if ($deletedOrgs->isEmpty()) {
            Log::info("No deleted organisations found.");
            return;
        }

        // Extract OrgIDs
        $orgIds = $deletedOrgs->pluck('OrgID')->filter()->values();

        Log::info("Found " . $orgIds->count() . " organisations marked as deleted.");

        // Map to local organisation IDs
        $localOrgIds = NhsOrganisation::whereIn('org_id', $orgIds)->pluck('id');

        if ($localOrgIds->isEmpty()) {
            Log::info("None of the deleted organisations exist locally.");
            return;
        }

        // Delete associated GP addresses
        $deletedCount = GpAddress::whereIn('nhs_organisation_id', $localOrgIds)->delete();

        Log::info("Deleted {$deletedCount} GP addresses linked to deleted organisations.");

        Log::info("Finished deleting GP addresses.");
    }
}
