<?php

namespace Mtc\GpAddresses\Http\Controllers;

use Mtc\GpAddresses\Models\GpAddress;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Illuminate\Http\Request;
use Mtc\GpAddresses\Http\Resources\NhsOrganisationResource;

class GpAddressesController extends Controller
{
    public function index()
    {
        return template('gpaddresses/index.twig');
    }

    public function fetch(Request $request)
    {
        $selectedCountry = $request->country;
        $selectedTown = $request->town;
        $selectedCounty = $request->county;
        $onlyActive = $request->onlyActive;

        $countries = GpAddress::query()
            ->distinct()
            ->orderBy('country')
            ->pluck('country');

        $counties = GpAddress::query()
            ->when($selectedCountry, function ($query) use ($selectedCountry) {
                $query->where('country', $selectedCountry);
            })
            ->whereNotNull('county')
            ->select('county')
            ->distinct()
            ->orderBy('county')
            ->pluck('county');

        $towns = GpAddress::query()
            ->when($selectedCounty, function ($query) use ($selectedCounty) {
                $query->where('county', $selectedCounty);
            })
            ->select('town')
            ->distinct()
            ->orderBy('town')
            ->pluck('town');

        // Get relationships
        $organisations = NhsOrganisation::with(['address', 'contact']);

        // Toggle active
        $organisations = $organisations->when($onlyActive, function ($query) {
            $query->where('status', 'Active');
        });

        // Filter by country and town
        $organisations = $organisations->whereHas('address', function ($query) use ($selectedCountry, $selectedTown) {
            $query->when($selectedCountry, function ($query) use ($selectedCountry) {
                $query->where('country', $selectedCountry);
            })
                ->when($selectedTown, function ($query) use ($selectedTown) {
                    $query->where('town', $selectedTown);
                });
        });

        // Handle search
        $organisations = $organisations->when($request->filled('search'), function ($query) use ($request) {
            $search = $request->search;

            // Search by org name, and address line
            $query->where(function ($query) use ($search) {
                $query->where('name', 'LIKE', "%{$search}%")
                    ->orWhereHas('address', function ($query) use ($search) {
                        $query->where('address_line_1', 'LIKE', "%{$search}%")
                            ->orWhere('address_line_2', 'LIKE', "%{$search}%")
                            ->orWhere('postcode', 'LIKE', "%{$search}");
                    });
            });
        });

        // Pagination
        $organisations = $organisations->paginate(20);

        return response()->json([
            'organisations' => NhsOrganisationResource::collection($organisations),
            'countries' => $countries,
            'towns' => $towns,
            'counties' => $counties,
            'current_page' => $organisations->currentPage(),
            'last_page' => $organisations->lastPage()
        ]);
    }
}
