<?php

namespace Mtc\GpAddresses\Services;

use Illuminate\Support\Facades\Http;
use Mtc\GpAddresses\Models\GpAddress;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Illuminate\Support\Facades\Log;
use Mtc\GpAddresses\Jobs\FetchAddresses;
use Mtc\GpAddresses\Jobs\FetchContacts;
use Illuminate\Support\Facades\DB;

class GpAddressesService
{

    private function commandLastRan () {
        $latestOrg = NhsOrganisation::orderBy('updated_at', 'desc')->first();

        $date = $latestOrg->updated_at->format('Y-m-d');

        return $date;
    }

    /**
     * Fetch all organisations from the NHS ODS API.
     * */
    private function fetchOrganisations($url, $initialFetch = false) {

        $attempts = 0;

        $orgs = [];

        while ( $url ) {
            do {
                $response = Http::get($url);

                $attempts++;

                Log::info("Fetching data from $url, attempt #$attempts, status {$response->status()}");

                if ($response->ok()) {
                    // Fetch records for Organisations from the NHS ODS API
                    $records = $response->json('Organisations') ?? [];

                    foreach( $records as $r ) {
                        if ( $initialFetch ) {
                            $org = [
                                'org_id' => $r['OrgId'] ?? null,
                                'name' => $r['Name'] ?? null,
                                'status' => $r['Status'] ?? null,
                            ];
                        }

                        $org = [
                            'org_link' => $r['OrgLink'] ?? null
                        ];

                        if ( in_array(null, $org, true) ) {
                            Log::warning("Organisation missing values. Skipping.");
                            continue;
                        }

                        $orgs[] = $org;
                    }

                    $url = $response->header('next-page') ?: null;
                    break; // success, exit retry loop
                }

            } while ($response->status() === 429 && $attempts < 5);
        }

        if ($attempts >= 5 && $response->status() === 429) {
            Log::warning("Max retries reached for $url. Stopping.");
            return;
        }

        return $orgs;
    }

    private function populateChildren($orgs) {
        $links = collect(array_column($orgs, 'org_link'));

        $batchSize = 5;

        $totalChunks = ceil($links->count() / $batchSize);

        $links->chunk($batchSize)->each(function ($chunk, $index) use ($totalChunks) {
            // Dispatch the jobs to job queue
            // Means app can get on with other tasks whilst we are still adding addresses
            Log::info("Dispatching jobs " . ($index + 1) . " of $totalChunks, chunk size: " . count($chunk));
            FetchAddresses::dispatch($chunk);
            FetchContacts::dispatch($chunk);
        });
    }

    /**
     * Fetch and store organisations in the database.
     *
     * @param $dispatchAddresses bool Option to dispatch jobs that can pull addresses in the background after we pull all the organisations.
     * @return void
     * */
    public function fetchAndStore(bool $dispatchAddresses = true): void
    {
        $url = config('gpaddresses.fetch') . '&Limit=1000';
        $attempts = 0;

        $orgs = $this->fetchOrganisations($url, true);

        NhsOrganisation::upsert(
            $orgs,
            // NOTE: Name was previous unqiue value, change this  back if this breaks
            ['org_id'], // unique key(s)
            ['name', 'status', 'org_link'] // columns to update if duplicate exists
        );

        $this->populateChildren($orgs);
    }

    public function sync($date = null): void
    {
        $url = config('gpaddresses.sync') . $date;
        $links = $this->fetchOrganisations($url);
        $this->populateChildren($links);
        $count = count($links);
        Log::info("Fetched a total of {$count} links.");
    }

    public function delete(): void {
        Log::info("Starting to delete GP addresses.");

        $url = config('gpaddresses.deleted');

        $response = Http::get($url);

        if (!$response->successful()) {
            Log::error("Failed to fetch deleted organisations: HTTP {$response->status()}");
            return;
        }

        // Handle downloaded file or normal JSON
        $raw = $response->body();
        $data = json_decode($raw, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            Log::error("Failed to decode JSON from response: " . json_last_error_msg());
            return;
        }

        $deletedOrgs = collect($data['DeletedOrganisations'] ?? []);

        if ($deletedOrgs->isEmpty()) {
            Log::info("No deleted organisations found.");
            return;
        }

        // Extract OrgIDs
        $orgIds = $deletedOrgs->pluck('OrgID')->filter()->values();

        Log::info("Found " . $orgIds->count() . " organisations marked as deleted.");

        // Map to local organisation IDs
        $localOrgIds = NhsOrganisation::whereIn('org_id', $orgIds)->pluck('id');

        if ($localOrgIds->isEmpty()) {
            Log::info("None of the deleted organisations exist locally.");
            return;
        }

        // Delete associated GP addresses
        $deletedCount = GpAddress::whereIn('nhs_organisation_id', $localOrgIds)->delete();

        Log::info("Deleted {$deletedCount} GP addresses linked to deleted organisations.");

        Log::info("Finished deleting GP addresses.");
    }
}
