<?php

namespace Mtc\GpAddresses\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\GpAddresses\Models\GpContactInformation;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Mtc\GpAddresses\Models\GpAddress;

class FetchGpInformation implements ShouldQueue
{
    private $links;

    use Queueable, Dispatchable;

    public function __construct($links)
    {
        $this->links = $links;
    }

    /**
     * Get Organisation Id from the database.
     *
     * @return id PK of Nhs Organisation.
     * */
    private function getOrganisationId($id)
    {
        return NhsOrganisation::where('org_id', $id)->value('id');
    }

    private function fetchAddresses($data)
    {
        $addresses = [];

        foreach ( $data as $item ) {
            $ext = $item['Organisation']['OrgId']['extension'] ?? null;
            $location = $item['Organisation']['GeoLoc']['Location'] ?? null;

            if (!$ext || !$location ) continue;

            $org = $this->getOrganisationId($ext);

            if (!$org) continue;

            $addresses[] = [
                    'nhs_organisation_id' => $org,
                    'address_line_1'      => $location['AddrLn1'] ?? null,
                    'postcode'            => $location['PostCode'] ?? null,
                    'uprn'                => $location['UPRN'] ?? null,
                    'address_line_2'      => $location['AddrLn2'] ?? null,
                    'town'                => $location['Town'] ?? null,
                    'county'              => $location['County'] ?? null,
                    'country'             => $location['Country'] ?? null,
            ];
        }

        if (!empty($addresses)) {
            GpAddress::upsert(
                $addresses,
                ['nhs_organisation_id', 'address_line_1', 'postcode'],
                ['uprn', 'address_line_2', 'town', 'county', 'country']
            );
        }
    }

    private function fetchContacts(array $data)
    {
        $contacts = [];

        foreach ($data as $item) {
            $ext = $item['Organisation']['OrgId']['extension'] ?? null;
            $contact = $item['Organisation']['Contacts']['Contact'] ?? [];

            if (!$ext || empty($contact)) continue;

            // Find the first telephone contact
            $telephone = $contact[0]['value'];

            if (!$telephone) continue;

            $orgId = $this->getOrganisationId($ext);
            if (!$orgId) continue;

            $contacts[] = [
                'nhs_organisation_id' => $orgId,
                'telephone_number'    => $telephone,
            ];
        }

        if ($contacts) {
            GpContactInformation::upsert(
                $contacts,
                ['nhs_organisation_id'], // unique key
                ['telephone_number']     // fields to update
            );
        }
    }

    public function handle(): void
    {
        $data = [];
        $attempts = 0;

        // Build the data.
        foreach ( $this->links as $link ) {
            $attempts = 0;

            do {
                $response = Http::retry(3, 10000)
                    ->get($link);

                $attempts++;

                if ( $response->ok() ) {
                    $data[] = $response->json();
                    break;
                }

                if ( $response->status() === 429 ) {
                    sleep(2);
                }

                Log::info("Attempted to fetch $link, after $attempts attempts with status {$response->status()}");

            } while( $response->status() === 429 && $attempts < 5);
        }

        // Process the data.
        $this->fetchAddresses($data);
        $this->fetchContacts($data);
    }
}
