<?php
/**
 * Class BaseImport
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 24.03.2017
 */

namespace Mtc\Import;

use Illuminate\Console\Command;
use Illuminate\Database\Capsule\Manager as Capsule;
use Mtc\Core\Images\ImageManipulation;

/**
 * Class BaseImport
 *
 * Implements base import functionality
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 24.03.2017
 */
abstract class BaseImport
{
    /**
     * IMPORT_CHUNK_SIZE - the size of chunks from DB to process to avoid memory issues during import
     */
    const IMPORT_CHUNK_SIZE = 1000;

    /**
     * @var Command $console
     */
    protected $console;

    /**
     * @var bool $truncate whether to truncate tables before importing records
     */
    protected $truncate = false;

    /**
     * @var bool $import_members whether to import member records
     */
    protected $import_members = false;

    /**
     * @var bool $import_categories whether to import categories
     */
    protected $import_categories = false;

    /**
     * @var bool $import_items whether to import item records
     */
    protected $import_items = false;

    /**
     * @var bool $import_item_images whether to import item images
     */
    protected $import_item_images = false;

    /**
     * @var array $image_folders Image folders, required if the images need to be imported
     */
    protected $image_folders = [];

    /**
     * @var string $old_site_url Old site URL with trailing slash
     */
    protected $old_site_url = '';

    /**
     * @var bool $import_orders whether to import order records
     */
    protected $import_orders = false;

    /**
     * @var bool $import_refunds whether to import refunds
     */
    protected $import_refunds = false;

    /**
     * @var bool $link_past_order_items whether to link order items with shop items or leave item ids as 0
     */
    protected $link_past_order_items = false;

    /**
     * @var bool $member_password_encrypted is member password encrypted in import db
     */
    protected $member_password_encrypted = true;

    /**
     * @var string $db_host remote DB host
     */
    protected $db_host;

    /**
     * @var string $db_user remote DB user
     */
    protected $db_user;

    /**
     * @var string $db_pass remote DB password
     */
    protected $db_pass;

    /**
     * @var string $db_name remote DB name
     */
    protected $db_name;

    /**
     * @var array Supported image types for image copying
     */
    protected static $supported_resize_types = [
        'jpg',
        'jpeg',
        'png'
    ];

    /**
     * Base Import constructor.
     *
     * Load the config values and set up local variables
     */
    public function __construct()
    {
        // Assign config values
        foreach (config('import.config', []) as $key => $value) {
            $this->{$key} = $value;
        }

        // Define connection
        $this->setRemoteConnection();
    }

    /**
     * Create a new connection for the remote database
     *
     * @param string $host DB host
     * @param string $username DB Username
     * @param string $password DB Password
     * @param string $db_name DB Name
     */
    public function setRemoteConnection()
    {
        $capsule = new Capsule(\Illuminate\Support\Facades\App::getInstance());
        $capsule->addConnection([
            'driver'    => 'mysql',
            'host'      => $this->db_host,
            'database'  => $this->db_name,
            'username'  => $this->db_user,
            'password'  => $this->db_pass,
            'charset'   => 'utf8',
            'collation' => 'utf8_unicode_ci',
            'prefix'    => '',
            'strict'    => false,
        ], 'import');
    }

    /**
     * Do the full import process based on config
     * This will trigger all relevant sub-imports based on given config
     */
    public function import(Command $console)
    {
        $this->console = $console;
        if (defined('MEMBERS_SECTION') && MEMBERS_SECTION === true && $this->import_members) {
            $this->importMembers();
        }

        if (defined('SHOP_ENABLED') && SHOP_ENABLED === true) {
            if ($this->import_categories) {
                $this->importCategories();
            }
            if ($this->import_items) {
                $this->importProducts();
            }
            if ($this->import_orders) {
                $this->importOrders();
            }
            if (defined('REFUNDS_ENABLED') && REFUNDS_ENABLED && $this->import_refunds) {
                $this->importRefunds();
            }
        }
    }

    /**
     * Function that copies an image file from server location and does all the cropping to distribute
     * the image if different sizes in the image folders
     *
     * Supported image types: see class variable $supported_resize_types
     *
     * @param string $old_location - Full path of file to be copied (can be a remote image, e.g., https://example.com/my_image.jpg)
     * @param string $new_name - New file name, including the extension, e.g., my_image.jpg
     * @param array $resize_params - Array of resize params for each image, e.g., $image_folders['product_folders']['mini']
     */

    public function copyImage($old_location, $new_name, $resize_params)
    {

        $new_file_path = SITE_PATH . '/' . $resize_params['path'] . '/' . $new_name;
        $extension = strtolower(end(explode('.', trim($new_file_path))));

        if (file_exists($new_file_path)) {
            unlink($new_file_path);
        }

        // Copies the file from one location to another
        copy($old_location, $new_file_path);

        // Do not continue if the image type is not supported for resizing.
        if (!in_array($extension, self::$supported_resize_types, true)) {
            return;
        }

        list($width, $height) = getimagesize($new_file_path);
        // Do not resize the image if it's the 'Originals' folder
        if ($resize_params['path'] === 'originals') {
            return;
        }

        // Do not resize if width and height smaller than resize dimensions
        if ($width <= $resize_params['width'] && $height <= $resize_params['height']) {
            return;
        }

        $resize_width = $width / $resize_params['width'];
        $resize_height = $height / $resize_params['height'];
        $ratio = ($resize_width > $resize_height) ? $resize_width : $resize_height;

        $new_width = ceil($width / $ratio);
        $new_height = ceil($height / $ratio);

        // Resize image
        if ($extension === 'png') {
            $source = imagecreatefrompng($new_file_path);
            $destination = imagecreatetruecolor($new_width, $new_height);
            $background = imagecolorallocate($destination , 0, 0, 0);
            imagecolortransparent($destination, $background);
            imagealphablending($destination, false);
            imagesavealpha($destination, true);
            imagecopyresampled($destination, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
            imagepng($destination, $new_file_path);
        } elseif (in_array($extension, ['jpg', 'jpeg'], true)) {
            $source = imagecreatefromjpeg($new_file_path);
            $destination = imagecreatetruecolor($new_width, $new_height);
            imagecopyresampled($destination, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
            imagejpeg($destination, $new_file_path, 100);
        }

        // Crop the image
        if ($resize_params['crop']) {
            ImageManipulation::imageCropCentercrop($resize_params['path'], $new_name, $resize_params['width'], $resize_params['height'], 0, 0);
        }

    }

}
