<?php

namespace Mtc\Import\Drivers;

use Illuminate\Support\Facades\DB;
use Mtc\Import\BaseImport;
use Mtc\Modules\Members\Classes\Member;
use Mtc\Modules\Members\Classes\MembersAddress;
use Mtc\Shop\Order;

/**
 * Class Import\MtcV1
 *
 * Import class for Mtc V1.
 * Allows importing information for product from a remote database.
 * Built to import content over from oldest Mtc platform
 *
 */
class MtcV1 extends BaseImport
{
    /**
     * Import Member information from WordPress
     *
     * Uses wp_users & wp_usermeta tables to gather information.
     * Will delete existing member records if Truncate is true
     */
    public function importMembers()
    {
        // Check if DB needs truncation
        if ($this->truncate) {
            $this->console->info("Truncating Members");
            Member::truncate();
            MembersAddress::truncate();
        }

        // Fetch All member entries from DB
        DB::connection('import')
            ->table('members')
            ->orderBy('id', 'asc')
            ->chunk(self::IMPORT_CHUNK_SIZE, function ($member_chunk) {

                $member_chunk->each(function ($member_record) {
                    /*
                     * Check if member already exists in database.
                     * This might happen if we skipped truncate and have already run import previously
                     * We don't need to create member record if it already exists.
                     * Member data update is not done.
                     */
                    if (!$this->truncate && Member::find($member_record->id)) {
                        return;
                    }

                    // Fetch user metadata
                    $member_addresses = DB::connection('import')->table('members_addresses')
                        ->where('member_id', $member_record->id)
                        ->get()
                        ->toArray();

                    /*
                     * Create a member record.
                     * We need to make sure correct member ID is passed through.
                     */
                    $member = new Member();
                    $member->id = $member_record->id;

                    // Some versions of Mtc v1 had password stored unencrypted, we can just move them over
                    if (!$this->member_password_encrypted) {
                        $member->setPasswordAttribute($member_record->password);
                    }

                    $member->fill([
                        'email' => $member_record->email,
                        'contact_no' => $member_record->contact_no,
                        'mobile_no' => $member_record->mobile_no,
                    ]);
                    $member->save();

                    foreach ($member_addresses as $address) {
                        $member->addresses()->create((array)$address);
                    }
                });
            });
        $this->console->info("Members Imported");

    }

    /**
     * Import Item information from old db
     *
     */
    public function importItems()
    {
        // TODO: Implement and submit to Bitbucket when you have to import items
    }

    /**
     * Import Order information from old db
     *
     */
    public function importOrders()
    {

        // Check if DB needs truncation
        if ($this->truncate) {
            $this->console->info("Truncating Orders");
            Order::truncate();
            Order\Address::truncate();
            Order\Info::truncate();
            Order\Item::truncate();
            Order\Coupon::truncate();
        }

        // Fetch All member entries from DB
        DB::connection('import')
            ->table('order')
            ->orderBy('id', 'asc')
            ->chunk(self::IMPORT_CHUNK_SIZE, function ($order_chunk) {
                $order_chunk->each(function ($order_record) {
                    /*
                     * Check if member already exists in database.
                     * This might happen if we skipped truncate and have already run import previously
                     * We don't need to create member record if it already exists.
                     * Member data update is not done.
                     */
                    if (!$this->truncate && Order::find($order_record->id)) {
                        return;
                    }


                    $order_items = DB::connection('import')
                        ->table('order_items')
                        ->where('order_id', $order_record->id)
                        ->get();

                    $order_info = DB::connection('import')
                        ->table('order_info')
                        ->where('order_id', $order_record->id)
                        ->get()
                        ->first();

                    $order_addresses = DB::connection('import')
                        ->table('order_address')
                        ->where('order_id', $order_record->id)
                        ->get()
                        ->toArray();

                    $order_coupon = DB::connection('import')
                        ->table('order_coupon')
                        ->where('order_id', $order_record->id)
                        ->get()
                        ->first();

                    $order_protx = DB::connection('import')
                        ->table('order_protx')
                        ->where('order_id', $order_record->id)
                        ->get()
                        ->first();

                    $order = new Order();
                    $order->id = $order_record->id;
                    $order->fill([
                        'paid' => 1,
                        'basket_id' => 0,
                        'date' => $order_record->date,
                        'delivery_name' => $order_record->delivery_name,
                        'delivery_cost' => $order_record->delivery_cost,
                        'coupon_code' => $order_coupon->code ?? '',
                        'cost_total' => $order_record->cost_total,
                        'cost_total_exvat' => $order_record->cost_total / (1 + VAT_RATE / 100),
                        'status' => $order_record->status,
                        'member' => $order_record->member,
                    ]);
                    $order->save();

                    $order->info()->create([
                        'email' => $order_info->email,
                        'contact_no' => $order_info->contact_no,
                    ]);

                    foreach ($order_addresses as $address) {
                        $order->address()->create((array)$address);
                    }

                    // So far we only support sagepay import.
                    DB::table('order_protx')
                        ->insert((array)$order_protx);

                    foreach ($order_items as $item) {
                        $order->items()->create([
                            'order_id' => $order_record->id,
                            'item_id' => $this->link_past_order_items ? $item->item_id : 0,
                            'item_name' => $item->item_name,
                            'item_price' => $item->item_price,
                            'quantity' => $item->quantity,
                            'size' => $item->size
                        ]);
                    }

                });
            });

        $this->console->info("Orders Imported");
    }

}