<?php

namespace Mtc\Import\Drivers;

use Illuminate\Database\Capsule\Manager as DB;
use Mtc\Import\BaseImport;
use Mtc\Modules\Members\Classes\Member;
use Mtc\Modules\Members\Classes\MembersAddress;
use Mtc\Plugins\Refunds\Classes\Refund;
use Mtc\Plugins\Refunds\Classes\RefundItem;
use Mtc\Shop\Category;
use Mtc\Shop\Item;
use Mtc\Shop\Order;

/**
 * Class Import\Wordpress
 *
 * WordPress import class.
 * Imports members and shop data (orders, products, categories, refunds)
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @version 04.06.2018
 */
class Wordpress extends BaseImport
{

    /**
     * @var array $order_status_map Maps Woocommerce order statuses to shop order statuses
     */
    private static $order_status_map = [
        'wc-approved'           => 3,
        'wc-completed'          => 4,
        'wc-cancelled'          => 9,
        'wc-refunded'           => 6,
        'wc-processing'         => 3,
        'wc-on-hold'            => 1,
    ];

    /**
     * Import Member information from WordPress
     *
     * Uses wp_users & wp_usermeta tables to gather information.
     * Will delete existing member records if Truncate is true
     */
    public function importMembers()
    {
        // Check if DB needs truncation
        if ($this->truncate) {
            $this->console->info("Truncating Members");
            Member::truncate();
            MembersAddress::truncate();
        }

        // Fetch All member entries from DB
        DB::connection('import')
            ->table('wp_users')
            ->get()
            ->each(function ($member_record) {
                /*
                 * Check if member already exists in database.
                 * This might happen if we skipped truncate and have already run import previously
                 * We don't need to create member record if it already exists.
                 * Member data update is not done.
                 */
                if (!$this->truncate && Member::find($member_record->ID)) {
                    return;
                }

                // Fetch user metadata
                $user_meta = DB::connection('import')->table('wp_usermeta')
                    ->where('user_id', $member_record->ID)
                    ->get()
                    ->keyBy('meta_key')
                    ->map(function ($meta) {
                        return $meta->meta_value;
                    })->toArray();

                /*
                 * Create a member record.
                 * We need to make sure correct member ID is passed through.
                 * WordPress Passwords are encrypted so we cannot migrate them over so we are leaving them empty.
                 */
                $member = new Member();
                $member->id = $member_record->ID;
                $member->fill([
                    'email' => $member_record->user_email,
                    'contact_no' => $user_meta['billing_phone'] ?? ''
                ]);
                $member->save();

                // Create users billing address
                $member->addresses()->create([
                    'type' => 'billing',
                    'firstname' => $user_meta['billing_first_name'] ?? '',
                    'lastname' => $user_meta['billing_last_name'] ?? '',
                    'address1' => $user_meta['billing_address_1'] ?? '',
                    'address2' => $user_meta['billing_address_2'] ?? '',
                    'city' => $user_meta['billing_city'] ?? '',
                    'state' => $user_meta['billing_state'] ?? '',
                    'country' => $user_meta['billing_country'] ?? '',
                    'postcode' => $user_meta['billing_postcode'] ?? ''
                ]);

                if (!empty($user_meta['shipping_country']) && !empty($user_meta['shipping_first_name'])) {
                    // Create users shipping address
                    $member->addresses()->create([
                        'type' => 'shipping',
                        'firstname' => $user_meta['shipping_first_name'] ?? '',
                        'lastname' => $user_meta['shipping_last_name'] ?? '',
                        'address1' => $user_meta['shipping_address_1'] ?? '',
                        'address2' => $user_meta['shipping_address_2'] ?? '',
                        'city' => $user_meta['shipping_city'] ?? '',
                        'state' => $user_meta['shipping_state'] ?? '',
                        'country' => $user_meta['shipping_country'] ?? '',
                        'postcode' => $user_meta['shipping_postcode'] ?? ''
                    ]);
                }
            });
        $this->console->info("Members Imported");
    }


    /**
     * Import Categories from WordPress
     * Deletes existing categories if $truncate is true
     */
    public function importCategories()
    {
        // Check if DB needs truncation
        if ($this->truncate) {
            $this->console->info("Truncating Categories");
            Category::query()->truncate();
        }

        $this->console->info("Importing categories...");
        DB::connection('import')
            ->table('wp_term_taxonomy')
            ->where('taxonomy', 'product_cat')
            ->join('wp_terms', 'wp_terms.term_id', '=', 'wp_term_taxonomy.term_id')
            ->get()
            ->each(function ($row) {
                $category = new Category();
                if ($this->truncate) {
                    $category->id = $row->term_id;
                } else {
                    $category->firstOrNew([
                        'id' => $row->term_id
                    ]);
                }

                $category->fill([
                    'sub_id' => $row->parent,
                    'name' => $row->name,
                    'slug' => $row->slug,
                    'description' => $row->description,
                ]);

                $category->save();
            });
        $this->console->info("Categories Imported");
    }

    /**
     * Import Products from WordPress
     * Deletes existing products and related data if $truncate is true
     *
     */
    public function importProducts()
    {
        // Check if DB needs truncation
        if ($this->truncate) {
            $this->console->info("Truncating Items Sizes Custom");
            Item\SizeCustom::query()->truncate();
            $this->console->info("Truncating Items Sizes");
            Item\Size::query()->truncate();
            $this->console->info("Truncating Items Custom");
            Item\Custom::query()->truncate();
            $this->console->info("Truncating Items Images");
            Item\Image::query()->truncate();
            $this->console->info("Truncating Items");
            Item::query()->truncate();
        }

        $this->console->info("Importing products...");
        // Fetch All product entries from DB
        DB::connection('import')
            ->table('wp_posts')
            ->where('post_type', 'product')
            ->get()
            ->each(function ($old_product) {

                // Fetch product metadata
                $product_meta = self::getPostMeta($old_product->ID);

                $product = new Item();
                // If truncated, no need to call DB
                if ($this->truncate) {
                    $product->id = $old_product->ID;
                } else {
                    $product->findOrNew([
                        'id' => $old_product->ID
                    ]);
                }

                if (!empty($product_meta['_price'])) {
                    if ($product_meta['_regular_price'] > 0) {
                        $price = (float)$product_meta['_regular_price'];
                    } else {
                        $price = (float)$product_meta['_price'];
                    }
                    $sale_price = (float)$product_meta['_sale_price'];
                } else {
                    if ($product_meta['_regular_price'] !== '') {
                        $price = (float)$product_meta['_regular_price'];
                        $sale_price = (float)$product_meta['_sale_price'];
                    } else {
                        $price = (float)$product_meta['_min_variation_regular_price'];
                        $sale_price = (float)$product_meta['_min_variation_sale_price'];
                    }
                }

                $stock = $product_meta['_stock'] ?? '';

                $product->fill([
                    'name' => $old_product->post_title,
                    'description' => $old_product->post_content,
                    'price' => (float)$price,
                    'price_exvat' => (float)$price,
                    'price_rrp' => (float)$price,
                    'sale_price' => (float)$sale_price,
                    'sale_price_exvat' => (float)$sale_price,
                    'sort_price' => (float)($sale_price > 0 ?: $price),
                    'hidden' => $old_product->post_status === 'publish' ? 0 : 1,
                    'stock' => $stock !== '' ?: ($product_meta['_stock_status'] === 'instock' ? 99999999 : 0),
                    'vat_rate' => 20,
                    'weight' => $product_meta['_weight'],
                    'vat_deductable' => 0,
                ]);
                $product->created_at = $old_product->post_date;
                $product->save();

                if ($product->custom()->count() === 0) {
                    $product->custom()->create([]);
                }

                // Item sizes
                DB::connection('import')
                    ->table('wp_posts')
                    ->where('post_type', 'product_variation')
                    ->where('post_parent', $old_product->ID)
                    ->get()
                    ->each(function ($old_size) {
                        // Fetch size metadata
                        $size_meta = self::getPostMeta($old_size->ID);

                        $size = new Item\Size();
                        // If truncated, no need to call DB
                        if ($this->truncate) {
                            $size->id = $old_size->ID;
                        } else {
                            $size->findOrNew([
                                'id' => $old_size->ID
                            ]);
                        }

                        $quantity = $size_meta['attribute_quantity'] ?? $size_meta['attribute_pa_quantity'] ?? '';
                        $strength = $size_meta['attribute_strength'] ?? $size_meta['attribute_pa_strength'] ?? '';
                        $size_value = trim($quantity . ' ' . $strength);

                        $stock = $size_meta['_stock'] ?? '';

                        $size->fill([
                            'item_id' => $old_size->post_parent,
                            'size' => $size_value,
                            'price' => (float)$size_meta['_regular_price'],
                            'price_exvat' => (float)$size_meta['_regular_price'],
                            'sale_price' => (float)($size_meta['_price'] < $size_meta['_regular_price'] ? $size_meta['_price'] : 0),
                            'sale_price_exvat' => (float)($size_meta['_price'] < $size_meta['_regular_price'] ? $size_meta['_price'] : 0),
                            'stock' => $stock !== '' ?: ($size_meta['_stock_status'] === 'instock' ? 99999999 : 0),
                            'hide' => $old_size->post_status === 'publish' ? 0 : 1,
                        ]);
                        $size->save();

                        // Item sizes custom

                        if ($size->custom()->count() === 0) {
                            $size->custom()->create([
                                'item_id' => $old_size->post_parent
                            ]);
                        }
                    });

                if (!$this->import_item_images || empty($this->old_site_url)) {
                    // Go to next item
                    return;
                }

                $old_image = DB::connection('import')
                    ->table('wp_posts')
                    ->where('ID', $product_meta['_thumbnail_id'])
                    ->first();
                // Imports item image
                if (!empty($old_image) && !empty($product_meta['_thumbnail_id'])) {
                    $image_meta = self::getPostMeta($old_image->ID);

                    $image = new Item\Image();

                    $image->findOrNew([
                        'item_id' => $old_product->ID
                    ]);

                    $image->fill([
                        'item_id' => $old_product->ID,
                        'name' => end(explode('/',  $image_meta['_wp_attached_file'])),
                        'default' => '1',
                    ]);
                    $image->save();
                    foreach ($this->image_folders['product_folders'] as $folder_name => $resize_params) {
                        $image_url = $this->old_site_url . 'wp-content/uploads/' . $image_meta['_wp_attached_file'];
                        $this->copyImage($image_url, $image->name, $resize_params);
                    }
                }

            });

        $this->console->info("Products Imported");

        if ($this->import_categories) {
            $this->importProductCategories();
        }
    }

    /**
     * Import Product categories from WordPress
     * Deletes existing Item categories if $truncate is true
     */
    public function importProductCategories()
    {
        // Check if DB needs truncation
        if ($this->truncate) {
            $this->console->info("Truncating Items Categories");
            DB::table('items_categories')->truncate();
        }

        $this->console->info("Importing product categories...");
        DB::connection('import')
            ->table('wp_term_relationships')
            ->join('wp_term_taxonomy', 'wp_term_taxonomy.term_taxonomy_id', '=', 'wp_term_relationships.term_taxonomy_id')
            ->where('taxonomy', 'product_cat')
            ->orderBy('parent')
            ->get()
            ->each(function ($row) {
                // If the table was truncated or the entry doesn't exist, insert it
                if ($item = (new Item())->find($row->object_id)) {
                    $item->categories()->syncWithoutDetaching($row->term_id);
                }
            });
        $this->console->info("Product Categories Imported");
    }

    /**
     * Import Orders from WordPress
     * Deletes existing orders and related data if $truncate is true
     *
     */
    public function importOrders()
    {
        // Check if DB needs truncation
        if ($this->truncate) {
            $this->console->info("Truncating Orders");
            Order::query()->truncate();
            $this->console->info("Truncating Order address");
            Order\Address::query()->truncate();
            $this->console->info("Truncating Order items");
            Order\Item::query()->truncate();
            $this->console->info("Truncating Order info");
            Order\Info::query()->truncate();
            $this->console->info("Truncating Order note");
            Order\Note::query()->truncate();
        }

        $this->console->info("Importing orders...");
        // Fetch All order entries from DB
        DB::connection('import')
            ->table('wp_posts')
            ->where('post_type', 'shop_order')
            ->whereIn('post_status', array_keys(self::$order_status_map))
            ->where('post_status', '!=', 'wc-cancelled')
            ->get()
            ->each(function ($old_order) {
                // Fetch order metadata
                $order_meta = self::getPostMeta($old_order->ID);

                // If order already exists, skip
                if (Order::query()->find($old_order->ID)) {
                    return true;
                }

                $order = new Order();

                $order->id = $old_order->ID;

                $delivery = DB::connection('import')
                    ->table('wp_woocommerce_order_items')
                    ->where('order_id', $old_order->ID)
                    ->where('order_item_type', 'shipping')
                    ->first();
                $delivery_meta = DB::connection('import')
                    ->table('wp_woocommerce_order_itemmeta')
                    ->where('order_item_id', $delivery->order_item_id)
                    ->get()
                    ->keyBy('meta_key')
                    ->map(function ($meta) {
                        return $meta->meta_value;
                    })->toArray();

                $status = self::getStatus($old_order->post_status);

                $order->fill([
                    'paid' => 1,
                    'date' => $old_order->post_date,
                    'delivery_name' => $delivery->order_item_name ?? '',
                    'delivery_cost' => $delivery_meta['cost'] ?? 0,
                    'cost_total' => $delivery_meta['cost'] ?? 0,
                    'cost_total_exvat' => $delivery_meta['cost'] ?? 0,
                    'status' => $status,
                    'member' => $order_meta['_customer_user'],
                    'order_ref' => $old_order->ID,
                ]);
                $order->save();

                // Order address
                $order->address()->delete();

                $order->address()->create([
                    'type' => 'billing',
                    'firstname' => $order_meta['_billing_first_name'] ?? '',
                    'lastname' => $order_meta['_billing_last_name'] ?? '',
                    'address1' => $order_meta['_billing_address_1'] ?? '',
                    'address2' => $order_meta['_billing_address_2'] ?? '',
                    'city' => $order_meta['_billing_city'] ?? '',
                    'state' => $order_meta['_billing_state'] ?? '',
                    'country' => $order_meta['_billing_country'] ?? '',
                    'postcode' => $order_meta['_billing_postcode'] ?? '',
                ]);

                $order->address()->create([
                    'type' => 'shipping',
                    'firstname' => $order_meta['_shipping_first_name'] ?? '',
                    'lastname' => $order_meta['_shipping_last_name'] ?? '',
                    'address1' => $order_meta['_shipping_address_1'] ?? '',
                    'address2' => $order_meta['_shipping_address_2'] ?? '',
                    'city' => $order_meta['_shipping_city'] ?? '',
                    'state' => $order_meta['_shipping_state'] ?? '',
                    'country' => $order_meta['_shipping_country'] ?? '',
                    'postcode' => $order_meta['_shipping_postcode'] ?? '',
                ]);

                // Order info
                $order->info()->delete();

                $order->info()->create([
                    'email' => $order_meta['_billing_email'],
                    'contact_no' => $order_meta['_billing_phone'],
                    'mobile_no' => $order_meta['_billing_phone'],
                    'newsletter' => 1,
                ]);

                // Order items

                $first = true;

                DB::connection('import')
                    ->table('wp_woocommerce_order_items')
                    ->where('order_id', $old_order->ID)
                    ->where('order_item_type', 'line_item')
                    ->get()
                    ->each(function ($old_order_item) use ($order, $order_meta, &$first) {
                        // Fetch order item metadata
                        $order_item_meta = DB::connection('import')
                            ->table('wp_woocommerce_order_itemmeta')
                            ->where('order_item_id', $old_order_item->order_item_id)
                            ->get()
                            ->keyBy('meta_key')
                            ->map(function ($meta) {
                                return $meta->meta_value;
                            })->toArray();

                        if ($order_item_meta['_line_total'] > 0) {

                            $order_item = new Order\Item();
                            // If truncated, no need to call DB
                            if ($this->truncate) {
                                $order_item->id = $old_order_item->order_item_id;
                            } else {
                                $order_item->findOrNew([
                                    'id' => $old_order_item->order_item_id
                                ]);
                            }

                            $product = (new Item)->find($order_item_meta['_product_id']);
                            $size = (new Item\Size)->find($order_item_meta['_variation_id']);

                            $price = $order_item_meta['_line_total'] / $order_item_meta['_qty'];

                            // Get discount per product. Although the discount is applied to order,
                            // we'll do this for the first item only, because we have discounts per items, now whole orders
                            $discount = 0;
                            if ($first) {
                                $discount = $order_meta['_cart_discount'] ? $order_meta['_cart_discount'] / $order_item_meta['_qty'] : 0;
                            }
                            $price_paid = $price - $discount;

                            $order_item->fill([
                                'order_id' => $old_order_item->order_id,
                                'item_id' => $order_item_meta['_product_id'] ?? 0,
                                'item_name' => $old_order_item->order_item_name ?? '',
                                'item_price' => $price,
                                'item_price_exvat' => $price,
                                'quantity' => $order_item_meta['_qty'],
                                'size' => !empty($size) ? $size->size : '',
                                'PLU' => !empty($product) ? $product->epos_code : '',
                                'discount' => 0,
                                'sizeid' => $order_item_meta['_variation_id'] ?? 0,
                                'vat_deductable' => 0,
                                'price_paid' => $price_paid,
                                'price_paid_exvat' => $price_paid,
                            ]);
                            $order_item->save();

                            // Update order totals
                            $order->cost_total += $order_item_meta['_line_total'];
                            $order->cost_total_exvat += $order_item_meta['_line_total'];
                            $order->save();

                            $first = false;
                        }
                    });

                // Order notes
                $order->notes()->delete();

                DB::connection('import')
                    ->table('wp_comments')
                    ->where('comment_post_ID', $old_order->ID)
                    ->get()
                    ->each(function ($old_order_note) use ($order) {
                        $order->notes()->create([
                            'note' => $old_order_note->comment_content,
                            'timestamp' => $old_order_note->comment_date,
                        ]);
                    });

                return true;
            });

        $this->console->info("Orders Imported");
    }

    /**
     * Imports refunds from Wordpress. Requires Refunds plugin: https://bitbucket.org/mtcmedia/core-plugins-refunds
     * Deletes existing refunds if $truncate is true
     */
    public function importRefunds()
    {
        // Check if DB needs truncation
        if ($this->truncate) {
            $this->console->info("Truncating Order Refunds");
            Refund::query()->truncate();
            $this->console->info("Truncating Order Refund Items");
            RefundItem::query()->truncate();
        }
        $this->console->info("Importing Refunds");

        DB::connection('import')
            ->table('wp_posts')
            ->where('post_type', 'shop_order_refund')
            ->where('post_status', '=', 'wc-completed')
            ->get()
            ->each(function ($old_refund) {
                $refund_meta = self::getPostMeta($old_refund->ID);

                $to_refund = (float)$refund_meta['_refund_amount'];

                if ($to_refund > 0) {
                    $order = (new Order())->find($old_refund->post_parent);

                    $refund = (new \Mtc\Plugins\Refunds\Classes\Refund())->create([
                        'order_id' => $order->id,
                        'delivery_refund_amount' => $order->delivery_cost,
                        'reference' => 'RF-' . $order->id,
                        'note' => '',
                        'created_at' => $old_refund->post_date,
                        'updated_at' => $old_refund->post_date,
                    ]);

                    // First refund all items
                    foreach ($order->items as $order_item) {

                        if ($to_refund > 0) {
                            $item_total = $order_item->price_paid * $order_item->quantity;
                            $item_refund = ($item_total - $to_refund > 0) ? $to_refund : $item_total;

                            $refund->items()->create([
                                'order_id' => $order->id,
                                'order_item_id' => $order_item->id,
                                'item_id' => $order_item->item_id,
                                'quantity' => $order_item->quantity,
                                'amount_refunded' => $item_refund,
                                'created_at' => $old_refund->post_date,
                                'updated_at' => $old_refund->post_date,
                            ]);

                            // Subtract the refunded value
                            $to_refund -= $item_refund;
                        }
                    }

                    // Eventually refund the delivery cost
                    $refund->delivery_refund_amount = ($order->delivery_cost - $to_refund > 0) ? $to_refund : $order->delivery_cost;
                    $refund->save();
                }
            });

        $this->console->info("Refunds Imported");
    }

    /**
     * Returns an array of post meta data keyed by meta field key.
     *
     * @param $post_id
     * @return array
     */
    private static function getPostMeta($post_id)
    {
        return DB::connection('import')
            ->table('wp_postmeta')
            ->where('post_id', $post_id)
            ->get()
            ->keyBy('meta_key')
            ->map(function ($meta) {
                return $meta->meta_value;
            })->toArray();
    }

    /**
     * Returns Shop order status from the WP status
     *
     * @param $wp_status
     * @return int|mixed
     */
    private static function getStatus($wp_status)
    {
        return self::$order_status_map[$wp_status] ?? 0;
    }


}