<?php

namespace Mtc\Insights;

use Mtc\Insights\Contracts\InsightMapInterface;

/**
 * Class PlatformMap
 *
 * @package Mtc\Insights
 */
class PlatformMap implements InsightMapInterface
{
    /**
     * Application base path
     *
     * @var string
     */
    protected $base_path;

    /**
     * PlatformMap constructor.
     */
    public function __construct()
    {
        $this->base_path = $base_path = dirname(__DIR__, 4);
    }

    /**
     * @inheritDoc
     */
    public function packages(): array
    {
        $composer_lock_contents = json_decode(file_get_contents($this->base_path . '/composer.lock'), true);
        $relevant_php_packages = $this->filterComposerLockPackages($composer_lock_contents['packages']);

        $package_json_contents = json_decode(file_get_contents($this->base_path . '/package.json'), true);
        $relevant_js_packages = $this->filterPackageJsonPackages($package_json_contents);

        return array_merge($relevant_php_packages, $relevant_js_packages);
    }

    /**
     * @inheritDoc
     */
    public function settings(): array
    {
        $settings = get_defined_constants(true);
        $list = [
            'php' => PHP_VERSION,
        ];

        foreach ($settings['user'] as $setting => $value) {
            if ($value === true && strpos($setting, 'TEST') === false) {
                $list[$setting] = '';
            }
        }

        return $list;
    }

    /**
     * @inheritDoc
     */
    public function components(): array
    {
        return array_merge(
            $this->mapComponents(glob($this->base_path . '/plugins/*')),
            $this->mapComponents(glob($this->base_path . '/modules/*'))
        );
    }

    /**
     * Map component names
     *
     * @param $full_path_array
     * @return array
     */
    protected function mapComponents($full_path_array)
    {
        $list = [];
        foreach ($full_path_array as $feature_path) {
            $list[str_replace($this->base_path . '/', '', $feature_path)] = '';
        }
        return $list;
    }

    /**
     * Filter the packages in composer lock to find relevant ones
     *
     * @param $packages
     * @return array
     */
    protected function filterComposerLockPackages($packages)
    {
        $list = [];
        foreach ($packages as $package) {
            if (strpos($package['name'], 'laravel/framework') !== false) {
                $list[$package['name']] = $package['version'];
            }

            if (strpos($package['name'], 'mtcmedia/') !== false) {
                $list[$package['name']] = $package['version'];
            }
        }

        return $list;
    }

    /**
     * Filter the packages in package.json to find relevant ones
     *
     * @param $packages
     * @return array
     */
    protected function filterPackageJsonPackages($packages)
    {
        $list = [];
        foreach ($packages['dependencies'] as $package => $version) {
            $list[$package] = $version;
        }
        foreach ($packages['devDependencies'] as $package => $version) {
            $list[$package] = $version;
        }

        return $list;
    }
}