<?php

namespace Mtc\Installer\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Mtc\Installer\Facades\Installer;

/**
 * Class CreateEnv
 *
 * @package Mtc\Installer\Console\Commands
 */
class CreateEnv extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'install:create-env';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create the environment information for site';

    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle()
    {
        $site_domain = $this->ask('App Url?');
        $site_name = $this->ask('Name of the application?');
        $site_id = Str::slug($site_name);
        $db_host = $this->ask('Database Host?', 'localhost');
        $db_name = $this->ask('Database Name?');
        $db_username = $this->ask('Database Username?');
        $db_password = $this->ask('Database Password?');

        $this->siteConfig();

        $this->info('Config stored. Please run `composer setup`');

        $this->setEnvFileValues([
            'site_name' => $site_name,
            'site_domain' => $site_domain,
            'site_id' => $site_id,
            'db_host' => $db_host,
            'db_name' => $db_name,
            'db_username' => $db_username,
            'db_password' => $db_password,
        ]);
    }

    /**
     * This config will be needed for the next scripts running so we store them in cache
     */
    protected function siteConfig()
    {
        Cache::forget('install-config');
        Cache::remember('install-config', now()->addHour(), function () {
            $bb_username = $this->ask('Bitbucket Username?');
            $bb_password = $this->secret('Bitbucket Password?');
            $dev_email = $this->ask('Developer email?');
            $theme = $this->chooseTheme();
            $root_modules = $this->chooseRootModules();
            if (array_key_exists('cms', $root_modules)) {
                $default_cms_setup = $this->confirm('Do you want to use default CMS page setup?', true);
            }
            $drop_git = $this->chooseDroppingGit();
            return [
                'bb_username' => $bb_username,
                'bb_password' => $bb_password,
                'dev_email' => $dev_email,
                'theme' => $theme,
                'root_modules' => $root_modules,
                'default_cms_setup' => $default_cms_setup ?? true,
                'drop_git' => $drop_git,
            ];
        });

    }

    /**
     * Choose which theme to install
     *
     * @return bool
     */
    protected function chooseTheme()
    {
        $themes = Installer::availableComponents('themes');
        $theme_names = $themes->pluck('name')
            ->push('other');

        $theme = $this->choice('Which Theme to install?', $theme_names->toArray(), $theme_names->first());
        if ($theme !== 'other') {
            $selected_theme = $themes->where('name', $theme)->first();
            return $selected_theme['full_name'];
        }

        $theme = $this->ask('Please specify repo name for the theme (e.g. mtcmedia/theme)?');
        $theme_url = Installer::repoUrl($theme);
        if ($this->confirm("Do you want to clone the theme from {$theme_url}?", true) === false) {
            return false;
        }
        return $theme_url;
    }


    /**
     * Choose which root modules to install
     *
     * @return Collection
     */
    protected function chooseRootModules()
    {
        return collect(config('installer.root_modules'))
            ->reject(function ($module) {
                // Core should already be installed
                return $module === 'core';
            })
            ->filter(function ($module, $module_path) {
                return $this->confirm("Do you want to install {$module_path}?");
            })->toArray();
    }

    /**
     * Check if user wants to drop git folders
     *
     * @return bool
     */
    protected function chooseDroppingGit()
    {
        return $this->confirm("Do you want to remove .git folders (are you working on a client site)?");
    }

    /**
     * Write info to .env file
     *
     * @param $data
     */
    protected function setEnvFileValues($data)
    {
        if (!File::exists(base_path('.env'))) {
            File::copy(base_path('.env.example'), base_path('.env'));
        }

        $env_file = File::get(base_path('.env'));
        $env_file = preg_replace([
            '/APP_NAME=(.*)/',
            '/APP_URL=(.*)/',
            '/SITE_ID=(.*)/',
            '/DB_HOST=(.*)/',
            '/DB_DATABASE=(.*)/',
            '/DB_USERNAME=(.*)/',
            '/DB_PASSWORD=(.*)/',
        ], [
            'APP_NAME="' . $data['site_name'] . '"',
            'APP_URL=' . $data['site_domain'],
            'SITE_ID=' . $data['site_id'],
            'DB_HOST=' . $data['db_host'],
            'DB_DATABASE=' . $data['db_name'],
            'DB_USERNAME=' . $data['db_username'],
            'DB_PASSWORD="' . $data['db_password'] . '"',
        ], $env_file);

        File::put(base_path('.env'), $env_file);

    }
}
