<?php

namespace Mtc\Installer\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Mtc\Installer\Facades\Installer;
use Mtc\Installer\InstallResponse;

/**
 * Class InstallSite
 *
 * @package App\Console\Commands
 */
class InstallSite extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'install:site';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Run the installation of a new site.';

    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle()
    {
        $config = Cache::get('install-config');
        if (!$config) {
            $this->error('Could not find config data. Please make sure you have run the environment setup');
            return;
        }

        $theme_installation = $this->installTheme($config);
        $this->installPlugins($theme_installation->dependencies());
        $this->installOptionalRootModules($config);
        $this->checkAndDropRepositories($config);

        if (!class_exists(\CreateJobsTable::class)) {
            $this->comment('Setting up Queue table');
            $this->call('queue:table');
        }

        try {
            $this->call('storage:link');
        } catch (\Exception $exception) {
            $this->info('Storage Linking failed, please check if symlink in /public/storage was created manually');
        }
    }

    /**
     * Install a theme
     *
     * @return mixed
     */
    protected function installTheme($config)
    {
        return $config['theme'] ? Installer::installTheme($config['theme']) : new InstallResponse(false);
    }

    /**
     * Loop through requirements and install plugins
     * This is done twice so dependencies of dependencies are also installed
     *
     * @param Collection $dependencies
     */
    protected function installPlugins($dependencies)
    {
        // make sure we don't have anything left from previously
        File::deleteDirectory(base_path('plugins'));
        File::deleteDirectory(base_path('modules'));

        $dependencies
            ->map(function ($url, $plugin_path) {
                $this->info("Installing {$plugin_path}");
                return Installer::installPlugin($url, $plugin_path)->dependencies();
            })
            ->filter()
            ->each(function ($sub_dependencies) use ($dependencies) {
                $sub_dependencies
                    ->reject(function ($url, $path) use ($dependencies) {
                        // Reject if already was part of first install phase
                        return !empty($dependencies[$path]);
                    })
                    ->each(function ($url, $plugin_path) {
                        $this->info("Installing {$plugin_path}");
                        return Installer::installPlugin($url, $plugin_path);
                    });
            });

        if (File::exists(base_path('plugins')) === false) {
            File::makeDirectory(base_path('plugins'));
        }

        if (File::exists(base_path('modules')) === false) {
            File::makeDirectory(base_path('modules'));
        }
    }

    /**
     * Install root modules cms/shop/abs
     */
    protected function installOptionalRootModules($config)
    {
        collect($config['root_modules'])
            ->each(function ($module, $module_path) {
                File::deleteDirectory(base_path($module_path));
                Installer::installRootModule($module, $module_path);
            });
    }

    /**
     * Check if this is a client site and .git repos should be dropped
     */
    protected function checkAndDropRepositories($config)
    {
        if ($config['drop_git']) {
            Installer::dropGitFolders();
        }
    }
}
