<?php

namespace Mtc\Installer;

use GitWrapper\GitWrapper;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Symfony\Component\Finder\SplFileInfo;

/**
 * Class Installer
 *
 * @package Mtc\Installer
 */
class Installer
{
    /**
     * @var GitWrapper
     */
    protected $git;

    /**
     * Installer constructor.
     *
     * @param GitWrapper $git
     */
    public function __construct(GitWrapper $git)
    {
        $this->git = $git;
    }

    /**
     * Install a root level module
     *
     * @param $module_name
     * @throws \Exception
     */
    public function installRootModule($module_name, $module_path): InstallResponse
    {
        if (!in_array($module_name, config('installer.root_modules'))) {
            throw new \Exception('Module not supported');
        }

        $module_url = $this->repoUrl("mtcmedia/{$module_name}");
        $result = $this->cloneRepository($module_url, base_path($module_path));
        return new InstallResponse($result->isCloned());
    }

    /**
     * Install a plugin on site
     *
     * @param $plugin_name
     * @param $plugin_path
     * @return InstallResponse
     */
    public function installPlugin($plugin_name, $plugin_path, bool $drop_git = false)
    {
        // requirements.php lists plugin full url, however we need to change that to add our auth
        $plugin_name = str_replace(['https://bitbucket.org/', '.git'], '', $plugin_name);
        $plugin_url = $this->repoUrl($plugin_name);
        $result = $this->cloneRepository($plugin_url, base_path($plugin_path));

        $dependencies = $this->getRequirementList("{$plugin_path}/admin/requirements.php");
        if ($drop_git && $result->isCloned() && File::isDirectory(base_path($plugin_path) . '/.git')) {
            File::deleteDirectory(base_path($plugin_path) . '/.git');
        }
        return new InstallResponse($result->isCloned(), $dependencies);
    }

    /**
     * Install a theme on site
     *
     * @param $theme_name
     * @return InstallResponse
     */
    public function installTheme($theme_name): InstallResponse
    {
        $site_id = Config::get('app.site_id');
        $module_url = $this->repoUrl($theme_name);
        File::deleteDirectory(base_path("sites/{$site_id}"));
        $result = $this->cloneRepository($module_url, base_path("sites/{$site_id}"));

        $dependencies = $this->getRequirementList("sites/{$site_id}/admin/requirements.php");
        return new InstallResponse($result->isCloned(), $dependencies);
    }

    /**
     * Find the list of available components
     *
     * @param $type
     * @return mixed
     */
    public function availableComponents($type)
    {
        $endpoint = config('installer.projects_api.endpoint') . '?' . http_build_query([
                'key' => \config('installer.projects_api.api_key'),
                'type' => $type
            ]);

        return Cache::remember("installer-components-{$type}", now()->addHour(), function () use ($endpoint) {
            $result = file_get_contents($endpoint);
            return collect(json_decode($result, true))
                ->reject(function ($component) {
                    return strpos($component['name'], 'Deprecated') !== false || $component['size'] < 59000;
                })
                ->map(function ($component, $key) {
                    $component['install_path'] = preg_replace('#^core/#', '', $component['name']);
                    $component['installed'] = File::exists(base_path($component['install_path']));
                    return $component;
                });
        });
    }

    /**
     * Remove git folders from site
     */
    public function dropGitFolders()
    {
        // Root level modules
        collect(config('installer.root_modules', []))
            ->each(function ($module_name, $module) {
                if (File::isDirectory(base_path("{$module}/.git"))) {
                    File::deleteDirectory(base_path("{$module}/.git"));
                }
            });

        // Theme
        $site_id = Config::get('app.site_id');
        if (File::isDirectory(base_path("sites/{$site_id}/.git"))) {
            File::deleteDirectory(base_path("sites/{$site_id}/.git"));
        }

        // Plugins & Modules
        collect(['plugins', 'modules'])
            ->each(function ($section) {
                collect(File::directories(base_path($section)))
                    ->each(function ($path_name) {
                        if (File::isDirectory($path_name . '/.git')) {
                            File::deleteDirectory($path_name . '/.git');
                        }
                    });
            });
    }

    /**
     * Get the bitbucket URL for cloning a repo
     *
     * @param $path_on_bitbucket
     * @return string
     */
    public function repoUrl($path_on_bitbucket)
    {
        $config = Cache::get('install-config');
        $username = $config['bb_username'];
        $password = $config['bb_password'];

        return "https://{$username}:{$password}@bitbucket.org/{$path_on_bitbucket}.git";
    }

    /**
     * Clone a repository
     *
     * @param $repository_name
     * @param $destination
     * @return \GitWrapper\GitWorkingCopy
     */
    protected function cloneRepository($repository_name, $destination)
    {
        return $this->git->cloneRepository($repository_name, $destination);
    }

    /**
     * Get the list of dependencies from requirement file (Legacy)
     *
     * @param $path
     * @return array
     */
    protected function getRequirementList($path)
    {
        $requirement_file_path = base_path($path);
        if (File::exists($requirement_file_path)) {
            // requirements.php will define plugins required variable
            include $requirement_file_path;
            return $plugins_required ?? [];
        }

        return [];
    }
}
