<?php
namespace Mtc\Klarna\Traits;

use Illuminate\Support\Collection;
use Mtc\Checkout\Invoice;
use Mtc\Klarna\LineType;

/**
 * Trait KlarnaInvoice
 * @package Mtc\Klarna\Traits
 */
trait KlarnaInvoice
{
    /**
     * Get Invoice data
     *
     * @return array
     */
    public function getInvoiceData(): array
    {
        $order_lines = $this->getOrderLinesFromBasket();

        $invoice_data = [
            'merchant_reference1' => strtoupper($this->invoice->payable_type) . '-' . $this->invoice->payable_id,
            'merchant_reference2' => 'INVOICE-' .$this->invoice_repository->getId(),
            'purchase_country' => $this->invoice->recipient->country,
            'purchase_currency' => $this->invoice_repository->getCurrency(),
            'locale' => config('app.locale'),
            'status' => self::COMPLETE,
            'auto_capture' => config('klarna.auto_capture', false),
            'order_amount' => $this->getMinorUnits(($this->invoice->use_ex_vat ? $this->invoice->outstanding_amount_ex_vat : $this->invoice->outstanding_amount)),
            'order_tax_amount' => $this->invoice->use_ex_vat ? 0 : collect($order_lines)->sum('total_tax_amount'),
            'order_lines' => $order_lines,
        ];

        if ($this->invoice->payable_type === 'order') {
            $invoice_data['shipping_address'] = $this->getAddress($this->invoice->payable->shippingAddress);
            $invoice_data['billing_address'] = $this->getAddress($this->invoice->recipient);
        }

        return $invoice_data;
    }

    /**
     * Get invoice capture data
     *
     * @return array
     */
    public function getInvoiceCaptureData(): array
    {
        return [
            'captured_amount' => $this->getMinorUnits(($this->invoice->use_ex_vat ? $this->invoice->outstanding_amount_ex_vat : $this->invoice->outstanding_amount)),
            'order_lines' => $this->getOrderLinesFromBasket()
        ];
    }

    /**
     * Get minor units
     *
     * @param $value
     * @return int
     */
    protected function getMinorUnits($value): int
    {
        return (int)round($value * 100);
    }

    /**
     * Get Line Type
     *
     * @param $type
     * @return string
     */
    protected function getLineType($type): string
    {
        if ($type === 'delivery') {
            return LineType::SHIPPING_FEE;
        }

        if ($type === 'coupon' || $type === 'discount') {
            return LineType::DISCOUNT;
        }

        if ($type === 'item') {
            return LineType::PHYSICAL;
        }

        if ($type === 'digital') {
            return LineType::DIGITAL;
        }

        return LineType::SURCHARGE;
    }

    /**
     * Get Order Lines From Basket
     *
     * @return array
     */
    protected function getOrderLinesFromBasket(): array
    {
        return collect($this->invoice->items)
            ->map(function(Invoice\Item $item) {
                $total_amount = $this->getMinorUnits($this->invoice->use_ex_vat ? $item->line_total_ex_vat->original() : $item->line_total->original());
                $vat_rate = is_numeric($item->vat_rate) ? $item->vat_rate : config('tax.vat_rates.' . $item->vat_rate);
                $tax_rate = $this->invoice->use_ex_vat ? 0 : $this->getMinorUnits($vat_rate * 100);

                return [
                    //Required
                    'name' => $item->name,
                    'quantity' => (int)$item->quantity,
                    'total_amount' => $total_amount,
                    'unit_price' => $this->getMinorUnits($this->invoice->use_ex_vat ? $item->original_value_ex_vat->original() : $item->original_value->original()),

                    //Optional
                    'tax_rate' => $tax_rate,
                    'total_tax_amount' => $this->invoice->use_ex_vat ? 0 : (int)round($total_amount - $total_amount * 10000 / (10000 + $tax_rate)),
                    'type' => $this->getLineType($item->parent_type === \Mtc\Orders\Item::class ? 'item' : 'digital'),
                    'reference' => optional($item->parent)->sku,
                ];
            })
            ->merge($this->getInvoiceLines())
            ->toArray();
    }

    /**
     * Get invoice lines
     *
     * @return Collection
     */
    public function getInvoiceLines(): Collection
    {
        return collect($this->invoice->lines)
            ->map(function(Invoice\Line $line) {
                $total_amount = $this->getMinorUnits($this->invoice->use_ex_vat ? $line->value_ex_vat : $line->value);
                $total_amount *= $line->is_negative ? -1 : 1;
                return [
                    'name' => $line->name,
                    'quantity' => 1,
                    'total_amount' => $total_amount,
                    'unit_price' => $total_amount,
                    'type' => $this->getLineType($line->type),
                ];
            });
    }

    /**
     * Get Address
     *
     * @param Collection $address
     * @return array|null
     */
    public function getAddress($address)
    {
        if ($address === null) {
            return null;
        }

        return [
            'email' => $this->invoice->email,
            'phone' => $this->invoice->contact_no,
            'given_name' => $address->first_name,
            'family_name' => $address->last_name,
            'street_address' => $address->address1,
            'street_address2' => $address->address2,
            'city' => $address->city,
            'country' => $address->country,
            'postal_code' => $address->postcode,
            'region' => $address->state ?: $address->county,
            'title' => $address->title,
            'organization_name' => null,
            'attention' => null
        ];
    }
}