<?php

namespace Mtc\Klarna;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\HasDeferredCharges;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Checkout\InvoiceRepository;
use Mtc\Checkout\PaymentForm;
use Mtc\Klarna\Contracts\KlarnaRepositoryContract;
use Mtc\Klarna\Repositories\KlarnaRepository;

/**
 * Class Klarna
 * @package Mtc\Klarna
 */
class Klarna implements PaymentGateway, HasDeferredCharges
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('klarna.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        /** @var KlarnaRepository $klarna */
        $klarna = App::make(KlarnaRepositoryContract::class, [
            'invoice_repository' => $invoice
        ]);

        return new PaymentForm('klarna-payment', 'vue-component', [
            'klarna' => $klarna->getInformation(),
            'pay_url' => route('charge_payment', [ $invoice->getId() ]) . '?gateway=klarna',
            'invoice_id' => $invoice->getId(),
            'name' => __('klarna::klarna.payment_option_name')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array
     * @throws Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        // Regular flow is handled in KlarnaController, only manual authorize flow is processed here
        if (config('klarna.auto_capture') === false) {
            return [];
        }

        if ($request->filled('authorization_token') === false) {
            throw new Exception('Authorization token is empty');
        }

        $klarna_order = App::make(KlarnaRepositoryContract::class, [
            'invoice_repository' => $invoice
        ])->placeOrder($request->input('authorization_token'));

        if ($klarna_order === false) {
            throw new Exception('Invalid Order details');
        }

        return [
            'provider' => 'klarna',
            'amount' => $invoice->getOutstandingAmount(),
            'amount_in_currency' => $klarna_order['original_order_amount'] / 100,
            'currency_code' => $invoice->getCurrency(),
            'reference' => $klarna_order['order_id'],
            'details' => $klarna_order,
            'confirmed_at' => config('klarna.auto_capture', false) === true ? now() : null,
        ];
    }

    /**
     * Check if payment can have a deferred charge
     *
     * @param Payment $payment
     * @return bool
     */
    public function allowDeferredCharge(Payment $payment): bool
    {
        if (config('checkout.deferred_payments') !== true) {
            return false;
        }

        return !config('klarna.auto_capture', false);
    }

    /**
     * Charge a payment that was set up as deferred
     *
     * @param Payment $payment
     * @return bool
     */
    public function chargeDeferredPayment(Payment $payment): bool
    {
        if (config('checkout.deferred_payments') !== true || config('klarna.auto_capture') === true) {
            return false;
        }

        /** @var InvoiceRepository $invoice_repository */
        $invoice_repository = App::make(InvoiceRepositoryContract::class);
        $invoice_repository->setModel($payment->invoice);

        App::make(KlarnaRepositoryContract::class, [
            'invoice_repository' => $invoice_repository
        ])->captureOrder($payment->reference);

        return true;
    }
}
