<?php

namespace Mtc\Klarna\Services;

use GuzzleHttp\Client;
use GuzzleHttp\RequestOptions;
use Illuminate\Support\Facades\Log;

/**
 * Class KlarnaApi
 * @package Mtc\Klarna\Services
 */
class KlarnaApi
{
    /**
     * Klarna API URL (Endpoint)
     *
     * @var string
     */
    protected $api_url;

    /**
     * KlarnaApi constructor.
     */
    public function __construct()
    {
        $this->api_url = $this->getApiUrl();
    }

    /**
     * Get Api URL
     *
     * @return string
     */
    protected function getApiUrl()
    {
        if (config('klarna.environment') === 'live') {
            return config('klarna.based_in') === 'US'
                ? 'https://api-na.klarna.com'
                : 'https://api.klarna.com';
        }

        return config('klarna.based_in') === 'US'
            ? 'https://api-na.playground.klarna.com'
            : 'https://api.playground.klarna.com';
    }

    /**
     * Call Klarna API
     *
     * @param $url
     * @param string $method
     * @param array $data
     * @return mixed|null
     */
    protected function callApi($url, $method = 'GET', $data = [])
    {
        try {
            $response = (new Client())->request($method, $url, [
                'auth' => [
                    config('klarna.credentials.username'),
                    config('klarna.credentials.password')
                ],
                RequestOptions::JSON => $data
            ]);

            return json_decode($response->getBody(), true);
        } catch (\GuzzleHttp\Exception\GuzzleException $exception) {
            Log::error('Klarna: ' . $exception->getMessage());
        } catch (\Exception $exception) {
            Log::error('Klarna: ' . $exception->getMessage());
        }

        return null;
    }

    /**
     * Create a new credit session
     *
     * @param array $data
     * @return mixed|null
     */
    public function createNewCreditSession(array $data)
    {
        return $this->callApi($this->api_url . '/payments/v1/sessions', 'POST', $data);
    }

    /**
     * Read an existing credit session
     *
     * @param $session_id
     * @return mixed|null
     */
    public function readCreditSession($session_id)
    {
        return $this->callApi($this->api_url . '/payments/v1/sessions/' . $session_id);
    }

    /**
     * Update an existing credit session
     *
     * @param $session_id
     * @param array $data
     * @return mixed|null
     */
    public function updateCreditSession($session_id, array $data)
    {
        return $this->callApi($this->api_url . '/payments/v1/sessions/' . $session_id, 'POST', $data);
    }

    /**
     * Create A New Order
     *
     * @param $tokern
     * @param array $data
     * @return mixed|null
     */
    public function createOrder($tokern, array $data)
    {
        return $this->callApi($this->api_url . '/payments/v1/authorizations/' . $tokern . '/order', 'POST', $data);
    }

    /**
     * Get order details
     *
     * @param $order_id
     * @return mixed|null
     */
    public function getOrder($order_id)
    {
        return $this->callApi($this->api_url . '/ordermanagement/v1/orders/' . $order_id);
    }

    /**
     * Cancel order
     *
     * @param $order_id
     * @return mixed|null
     */
    public function cancelOrder($order_id)
    {
        return $this->callApi($this->api_url . '/ordermanagement/v1/orders/' . $order_id . '/cancel', 'POST');
    }

    /**
     * Create capture
     *
     * @param $order_id
     * @param array $data
     * @return mixed|null
     */
    public function createCapture($order_id, array $data)
    {
        return $this->callApi($this->orderEndpoint($order_id . '/captures'), 'POST', $data);
    }

    /**
     * Update merchant references (merchant_reference1, merchant_reference2)
     *
     * @param $order_id
     * @param array $data
     * @return mixed|null
     */
    public function updateMerchantReferences($order_id, array $data)
    {
        return $this->callApi($this->orderEndpoint($order_id . '/merchant-references'), 'PATCH', $data);
    }

    /**
     * Generate a consumer token
     *
     * @param $tokern
     * @param array $data
     * @return mixed|null
     */
    public function generateConsumerToken($tokern, array $data)
    {
        return $this->callApi(
            $this->api_url . '/payments/v1/authorizations/' . $tokern . '/customer-token',
            'POST',
            $data
        );
    }

    /**
     * Cancel an existing authorization
     *
     * @param $tokern
     * @return mixed|null
     */
    public function cancelAuthorization($tokern)
    {
        return $this->callApi($this->api_url . '/payments/v1/authorizations/' . $tokern, 'DELETE');
    }

    /**
     * Create a refund
     *
     * @param $order_id
     * @param array $data
     * @return mixed|null
     */
    public function createRefund($order_id, array $data)
    {
        return $this->callApi($this->orderEndpoint($order_id . '/refunds'), 'POST', $data);
    }

    /**
     * Get refund
     *
     * @param $order_id
     * @param $refund_id
     * @return mixed|null
     */
    public function getRefund($order_id, $refund_id)
    {
        return $this->callApi($this->orderEndpoint($order_id . '/refunds/' . $refund_id));
    }

    /**
     * Get url to order management endpoint
     *
     * @param string $path
     * @return string
     */
    protected function orderEndpoint(string $path): string
    {
        return $this->api_url . '/ordermanagement/v1/orders/' . $path;
    }
}
