<?php
namespace Mtc\Clans\Model\ResourceModel;

use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Psr\Log\LoggerInterface;
use Mtc\Clans\Api\Data\ClanManagerInterface;
use Magento\Framework\EntityManager\EntityManager;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Backend\Helper\Js;

/**
 * Main class to load data from db
 *
 * Class ClanManager
 */
class ClanManager extends AbstractDb
{
    /**
     * @var EntityManager
     */
    protected $entityManager;

    /**
     * @var MetadataPool
     */
    protected $metadataPool;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var Js
     */
    private $helperJs;

    /**
     * @param Context $context
     * @param LoggerInterface $logger
     * @param EntityManager $entityManager
     * @param MetadataPool $metadataPool
     * @param Js $helperJs
     * @param null $connectionName
     */
    public function __construct(
        Context $context,
        LoggerInterface $logger,
        EntityManager $entityManager,
        MetadataPool $metadataPool,
        Js $helperJs,
        $connectionName = null
    ) {
        $this->logger = $logger;
        $this->entityManager = $entityManager;
        $this->metadataPool = $metadataPool;
        $this->helperJs = $helperJs;
        parent::__construct($context, $connectionName);
    }

    /**
     * Resource initialization
     * @return void
     */
    protected function _construct()
    {
        $this->_init('mtc_clan', 'entity_id');
    }

    /**
     * @inheritDoc
     */
    public function getConnection()
    {
        return $this->metadataPool->getMetadata(ClanManagerInterface::class)->getEntityConnection();
    }

    /**
     * @param $surnameId
     * @return array
     */
    public function getClansBySurnameId($surnameId)
    {
        $blockTable = $this->getTable('mtc_surname_clan_link');
        $adapter = $this->getConnection();
        $select = $adapter->select()
            ->from(['surnamelink' => $blockTable])
            ->where($adapter->prepareSqlCondition('surnamelink.surname_id', ['in' => [$surnameId]]));
        return $adapter->fetchAll($select);
    }

    /**
     * @param AbstractModel $object
     * @return $this
     */
    public function save(AbstractModel $object)
    {
        try {
            $this->entityManager->save($object);
            if (!$object->getClanId()) {
                $object->setClanId($object->getEntityId());
                $this->entityManager->save($object);
            }
            $links = $object['links'];
            if (isset($links['surname'])) {
                $condition = $this->getConnection()
                    ->quoteInto('clan_id = ?', $object->getId());

                $blockIds = $this->helperJs->decodeGridSerializedInput($links['surname']);
                $this->getConnection()
                    ->delete($this->getTable('mtc_surname_clan_link'), $condition);

                foreach ($blockIds as $id) {
                    $blocks = [];
                    $blocks['clan_id'] = $object->getId();
                    $blocks['surname_id'] = $id;
                    $this->getConnection()
                        ->insert($this->getTable('mtc_surname_clan_link'), $blocks);
                }
            }
            if (isset($links['tartan'])) {
                $condition = $this->getConnection()
                    ->quoteInto('clan_id = ?', $object->getId());

                $blockIds = $this->helperJs->decodeGridSerializedInput($links['tartan']);
                $this->getConnection()
                    ->delete($this->getTable('mtc_clan_tartan_link'), $condition);

                foreach ($blockIds as $id) {
                    $blocks = [];
                    $blocks['clan_id'] = $object->getId();
                    $blocks['tartan_id'] = $id;
                    $this->getConnection()
                        ->insert($this->getTable('mtc_clan_tartan_link'), $blocks);
                }
            }
            return $this;
        } catch (\Exception $exception) {
            $this->logger->critical($exception->getMessage());
        }
    }

    /**
     * @param AbstractModel $object
     * @return $this|ClanManager
     */
    public function delete(AbstractModel $object)
    {
        try {
            $this->entityManager->delete($object);
            return $this;
        } catch (\Exception $exception) {
            $this->logger->debug($exception->getMessage());
        }
    }
}
