<?php
namespace Mtc\Clans\ViewModel;

use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Filesystem;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Mtc\Clans\Api\ClanManagerRepositoryInterface;
use Mtc\Clans\Api\SurnameManagerRepositoryInterface;
use Psr\Log\LoggerInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\Api\FilterBuilder;
use Magento\Catalog\Model\CategoryRepository;
use Magento\Framework\App\Request\Http;
use Mtc\Clans\Model\ResourceModel\ClanManager;
use Mtc\Clans\Model\ClanManagerFactory as ClanManagerModel;

/**
 * All clans view model
 */
class ClansList implements ArgumentInterface
{
    /**
     * @var ClanManagerRepositoryInterface
     */
    private $clanManagerRepository;

    /**
     * @var SurnameManagerRepositoryInterface
     */
    private $surnameManagerRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var FilterBuilder
     */
    private $filterBuilder;

    /**
     * @var CategoryRepository
     */
    private $categoryRepository;

    /**
     * @var Http
     */
    private $request;

    /**
     * @var ClanManager
     */
    private $clanManager;

    /**
     * @var ClanManagerModel
     */
    private $clanManagerModel;

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * @var File
     */
    private $fileDriver;

    /**
     * @param ClanManagerRepositoryInterface $clanManagerRepository
     * @param LoggerInterface $logger
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param StoreManagerInterface $storeManager
     * @param FilterBuilder $filterBuilder
     * @param CategoryRepository $categoryRepository
     * @param Http $request
     * @param SurnameManagerRepositoryInterface $surnameManagerRepository
     * @param ClanManager $clanManager
     * @param ClanManagerModel $clanManagerModel
     * @param File $fileDriver
     * @param Filesystem $filesystem
     */
    public function __construct(
        ClanManagerRepositoryInterface $clanManagerRepository,
        LoggerInterface $logger,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoreManagerInterface $storeManager,
        FilterBuilder $filterBuilder,
        CategoryRepository $categoryRepository,
        Http $request,
        SurnameManagerRepositoryInterface $surnameManagerRepository,
        ClanManager $clanManager,
        ClanManagerModel $clanManagerModel,
        File $fileDriver,
        Filesystem $filesystem
    ) {
        $this->clanManagerRepository = $clanManagerRepository;
        $this->logger = $logger;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storeManager = $storeManager;
        $this->filterBuilder = $filterBuilder;
        $this->categoryRepository = $categoryRepository;
        $this->request = $request;
        $this->surnameManagerRepository = $surnameManagerRepository;
        $this->clanManager = $clanManager;
        $this->clanManagerModel = $clanManagerModel;
        $this->fileDriver = $fileDriver;
        $this->filesystem = $filesystem;
    }

    /**
     * @return ClanManagerRepositoryInterface
     */
    public function getAllClans() {
        $search = $this->request->getParam('search');
        $query = $this->request->getParam('query');
        if ($search) {
            $filters[] = $this->filterBuilder
                ->setField('clan')
                ->setConditionType('like')
                ->setValue($search.'%')
                ->create();
            $this->searchCriteriaBuilder->addFilters($filters);
            $searchCriteria = $this->searchCriteriaBuilder->create();
            return $this->clanManagerRepository->getList($searchCriteria);
        }
        if ($query) {
            $surnamesList = null;
            $filters[] = $this->filterBuilder
                ->setField('surname')
                ->setConditionType('like')
                ->setValue($query.'%')
                ->create();
            $this->searchCriteriaBuilder->addFilters($filters);
            $searchCriteria = $this->searchCriteriaBuilder->create();
            $surnamesList = $this->surnameManagerRepository->getList($searchCriteria);
            $clanIds = [];
            $surnameIds = [];
            if ($surnamesList && $surnamesList->getTotalCount()) {
                foreach ($surnamesList->getItems() as $surname) {
                    $surnameIds[] = $surname->getEntityId();
                }
            }
            if ($surnameIds) {
                foreach ($surnameIds as $surnameId) {
                    $clansData = $this->clanManager->getClansBySurnameId($surnameId);
                    foreach ($clansData as $clan) {
                        $clanIds[] = $clan['clan_id'];
                    }
                }
            }
            if ($clanIds) {
                $searchCriteria = $this->searchCriteriaBuilder
                    ->addFilter('entity_id', $clanIds, 'in')
                    ->create();
                return $this->clanManagerRepository->getList($searchCriteria);
            }
        }
        return null;
    }

    /**
     * @return mixed
     */
    public function getSearchParam() {
        return $this->request->getParam('search');
    }

    /**
     * @return mixed
     */
    public function getQueryParam() {
        return $this->request->getParam('query');
    }

    /**
     * @return mixed
     */
    public function getClan() {
        $clanId =  $this->request->getParam('clan');
        if ($clanId) {
            try{
                return $this->clanManagerRepository->getById($clanId);
            } catch (\Exception $e) {
                $this->logger->critical($e->getMessage());
            }
        }
        return null;
    }

    /**
     * @return null
     */
    public function getTartans() {
        $clanId =  $this->request->getParam('clan');
        if ($clanId) {
            $model = $this->clanManagerModel->create();
            $tartans = $model->getTartans($clanId);
            if (is_array($tartans) and count($tartans)) {
                return $tartans;
            }
        }
        return null;
    }

    /**
     * It will return the thumbanil image URL
     *
     * @return string
     * @throws FileSystemException
     */
    public function getTartanImageUrl($file)
    {
        if ($file == '') {
            return '';
        }
        $mediapath = $this->filesystem->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath();
        if (!$this->fileDriver->isExists($mediapath.'/'.$file)) {
            $file = 'TartanManager/awaiting-image.jpg';
        }
        return $this->getMediaUrl($file);
    }

    /**
     * Get file url
     * @param $file
     * @return string
     */
    private function getMediaUrl($file)
    {
        try {
            $file = ltrim(str_replace('\\', '/', $file), '/');
            return $this->storeManager
                    ->getStore()
                    ->getBaseUrl(UrlInterface::URL_TYPE_MEDIA) . $file;
        } catch (\Exception $exception) {
            $this->logger->critical($exception->getMessage());
        }
    }

    /**
     * @param $categoryId
     * @param $tartanId
     * @return string|null
     */
    public function getCategoryUrl($categoryId, $clanId) {
        try {
            $params = ['clan' => $clanId];
            $urlParams = [];
            foreach ($params as $name=>$value){
                $urlParams[] = $name.'='.urlencode($value);
            }
            $urlParams = implode('&', $urlParams);
            $categoryUrl = $this->categoryRepository->get($categoryId, $this->storeManager->getStore()->getId())->getUrl();
            if ($urlParams){
                $glue = '?';
                if (str_contains($categoryUrl, $glue)){
                    $glue = '&';
                }
                $categoryUrl .= $glue.$urlParams;
            }
            return $categoryUrl;
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }
        return null;
    }
}
