<?php
namespace Mtc\Clans\Model;

use Mtc\Clans\Api\ClanManagerRepositoryInterface;
use Mtc\Clans\Api\Data\ClanManagerSearchResultsInterface;
use Mtc\Clans\Model\ClanManager as ClanManagerAlias;
use Mtc\Clans\Model\ResourceModel\ClanManager as ResourceClanManager;
use Mtc\Clans\Model\ResourceModel\ClanManager\CollectionFactory as ClanManagerCollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Mtc\Clans\Api\Data\ClanManagerInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Mtc\Clans\Model\ResourceModel\ClanManager\Collection;
use Mtc\Clans\Api\Data\ClanManagerSearchResultsInterfaceFactory;
use Psr\Log\LoggerInterface;

/**
 * This is ClanManagerRepository class
 * Class ClanManagerRepository
 */
class ClanManagerRepository implements ClanManagerRepositoryInterface
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ResourceClanManager
     */
    private $resourceEventManager;

    /**
     * @var ClanManagerFactory
     */
    private $eventManagerFactory;

    /**
     * @var ClanManagerCollectionFactory
     */
    private $eventManagerCollectionFactory;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var ClanManagerSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @param ResourceClanManager $resourceEventManager
     * @param ClanManagerFactory $eventManagerFactory
     * @param ClanManagerCollectionFactory $eventManagerCollectionFactory
     * @param ClanManagerSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface|null $collectionProcessor
     * @param LoggerInterface $logger
     */
    public function __construct(
        ResourceClanManager $resourceEventManager,
        ClanManagerFactory $eventManagerFactory,
        ClanManagerCollectionFactory $eventManagerCollectionFactory,
        ClanManagerSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor = null,
        LoggerInterface $logger
    ) {
        $this->resourceEventManager = $resourceEventManager;
        $this->eventManagerFactory = $eventManagerFactory;
        $this->eventManagerCollectionFactory = $eventManagerCollectionFactory;
        $this->collectionProcessor = $collectionProcessor;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->logger = $logger;
    }

    /**
     * Save Block data
     * @param ClanManagerInterface $eventManager
     * @return ClanManagerInterface
     * @throws CouldNotSaveException
     */
    public function save(ClanManagerInterface $eventManager)
    {
        try {
            $this->resourceEventManager->save($eventManager);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }
        return $eventManager;
    }

    /**
     * Load Block data by given Block Identity
     *
     * @param $eventManagerId
     * @return ClanManagerAlias
     */
    public function getById($eventManagerId)
    {
        /**
         * @var ClanManager $eventManager
         */
        $eventManager = $this->eventManagerFactory->create();
        $this->resourceEventManager->load($eventManager, $eventManagerId);
        return $eventManager;
    }

    /**
     * @param $clanId
     * @return null
     */
    public function getEntityIdByClanId($clanId)
    {
        $customer = $this->eventManagerCollectionFactory->create();
        $dataUser = $customer->addFieldToFilter('clan_id', $clanId)
            ->getFirstItem();
        if ($dataUser && $dataUser->getId()) {
            return $dataUser->getEntityId();
        } else {
            return null;
        }
    }

    /**
     * Load Page data collection by given search criteria
     *
     * @param SearchCriteriaInterface $criteria
     * @return ClanManagerSearchResultsInterface
     */

    public function getList(SearchCriteriaInterface $criteria)
    {
        /** @var Collection $collection */
        $collection = $this->eventManagerCollectionFactory->create();

        foreach ($criteria->getFilterGroups() as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                $condition = $filter->getConditionType() ?: 'eq';
                $collection->addFieldToFilter($filter->getField(), [$condition => $filter->getValue()]);
            }
        }

        $this->collectionProcessor->process($criteria, $collection);

        /** @var ClanManagerSearchResultsInterface $searchResults */
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * Delete Block
     *
     * @param ClanManagerInterface $eventManager
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function delete(ClanManagerInterface $eventManager)
    {
        try {
            $this->resourceEventManager->delete($eventManager);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * Delete ClanManager by given Block Identity
     *
     * @param string $eventManagerId
     * @return bool|mixed
     */
    public function deleteById($eventManagerId)
    {
        try {
            return $this->delete($this->getById($eventManagerId));
        } catch (\Exception $exception) {
            $this->logger->debug($exception->getMessage());
        }
    }
}
