<?php
namespace Mtc\ClansEvents\Model;

use Mtc\ClansEvents\Api\EventManagerRepositoryInterface;
use Mtc\ClansEvents\Api\Data\EventManagerSearchResultsInterface;
use Mtc\ClansEvents\Model\EventManager as EventManagerAlias;
use Mtc\ClansEvents\Model\ResourceModel\EventManager as ResourceEventManager;
use Mtc\ClansEvents\Model\ResourceModel\EventManager\CollectionFactory as EventManagerCollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Mtc\ClansEvents\Api\Data\EventManagerInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Mtc\ClansEvents\Model\ResourceModel\EventManager\Collection;
use Mtc\ClansEvents\Api\Data\EventManagerSearchResultsInterfaceFactory;
use Psr\Log\LoggerInterface;

class EventManagerRepository implements EventManagerRepositoryInterface
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ResourceEventManager
     */
    private $resourceEventManager;

    /**
     * @var EventManagerFactory
     */
    private $eventManagerFactory;

    /**
     * @var EventManagerCollectionFactory
     */
    private $eventManagerCollectionFactory;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var EventManagerSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @param ResourceEventManager $resourceEventManager
     * @param EventManagerFactory $eventManagerFactory
     * @param EventManagerCollectionFactory $eventManagerCollectionFactory
     * @param EventManagerSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface|null $collectionProcessor
     * @param LoggerInterface $logger
     */
    public function __construct(
        ResourceEventManager $resourceEventManager,
        EventManagerFactory $eventManagerFactory,
        EventManagerCollectionFactory $eventManagerCollectionFactory,
        EventManagerSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor = null,
        LoggerInterface $logger
    ) {
        $this->resourceEventManager = $resourceEventManager;
        $this->eventManagerFactory = $eventManagerFactory;
        $this->eventManagerCollectionFactory = $eventManagerCollectionFactory;
        $this->collectionProcessor = $collectionProcessor;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->logger = $logger;
    }

    /**
     * @param EventManagerInterface $eventManager
     * @return EventManagerInterface
     * @throws CouldNotSaveException
     */
    public function save(EventManagerInterface $eventManager)
    {
        try {
            $this->resourceEventManager->save($eventManager);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }
        return $eventManager;
    }

    /**
     * @param $eventManagerId
     * @return EventManagerAlias
     */
    public function getById($eventManagerId)
    {
        /**
         * @var EventManager $eventManager
         */
        $eventManager = $this->eventManagerFactory->create();
        $this->resourceEventManager->load($eventManager, $eventManagerId);
        return $eventManager;
    }

    /**
     * @param SearchCriteriaInterface $criteria
     * @return EventManagerSearchResultsInterface
     */

    public function getList(SearchCriteriaInterface $criteria)
    {
        /** @var Collection $collection */
        $collection = $this->eventManagerCollectionFactory->create();

        foreach ($criteria->getFilterGroups() as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                $condition = $filter->getConditionType() ?: 'eq';
                $collection->addFieldToFilter($filter->getField(), [$condition => $filter->getValue()]);
            }
        }

        $this->collectionProcessor->process($criteria, $collection);

        /** @var EventManagerSearchResultsInterface $searchResults */
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * @param EventManagerInterface $eventManager
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function delete(EventManagerInterface $eventManager)
    {
        try {
            $this->resourceEventManager->delete($eventManager);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @param string $eventManagerId
     * @return bool|mixed
     */
    public function deleteById($eventManagerId)
    {
        try {
            return $this->delete($this->getById($eventManagerId));
        } catch (\Exception $exception) {
            $this->logger->debug($exception->getMessage());
        }
    }
}
