<?php
namespace Mtc\Finder\Model;

use Mtc\Finder\Api\Data\TartanManagerSearchResultsInterface;
use Mtc\Finder\Api\TartanManagerRepositoryInterface;
use Mtc\Finder\Model\TartanManager as TartanManagerAlias;
use Mtc\Finder\Model\ResourceModel\TartanManager as ResourceTartanManager;
use Mtc\Finder\Model\ResourceModel\TartanManager\CollectionFactory as TartanManagerCollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Mtc\Finder\Api\Data\TartanManagerInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Mtc\Finder\Model\ResourceModel\TartanManager\Collection;
use Mtc\Finder\Api\Data\TartanManagerSearchResultsInterfaceFactory;
use Psr\Log\LoggerInterface;

/**
 * This is TartanManagerRepository class
 * Class TartanManagerRepository
 */
class TartanManagerRepository implements TartanManagerRepositoryInterface
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ResourceTartanManager
     */
    private $resourceEventManager;

    /**
     * @var TartanManagerFactory
     */
    private $eventManagerFactory;

    /**
     * @var TartanManagerCollectionFactory
     */
    private $eventManagerCollectionFactory;

    /**
     * @var TartanManagerInterface
     */
    private $dataEventManager;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var TartanManagerSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @param ResourceTartanManager $resourceEventManager
     * @param TartanManagerFactory $eventManagerFactory
     * @param TartanManagerCollectionFactory $eventManagerCollectionFactory
     * @param TartanManagerSearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface|null $collectionProcessor
     * @param LoggerInterface $logger
     */
    public function __construct(
        ResourceTartanManager $resourceEventManager,
        TartanManagerFactory $eventManagerFactory,
        TartanManagerCollectionFactory $eventManagerCollectionFactory,
        TartanManagerSearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor = null,
        LoggerInterface $logger
    ) {
        $this->resourceEventManager = $resourceEventManager;
        $this->eventManagerFactory = $eventManagerFactory;
        $this->eventManagerCollectionFactory = $eventManagerCollectionFactory;
        $this->collectionProcessor = $collectionProcessor;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->logger = $logger;
    }

    /**
     * Save Block data
     * @param TartanManagerInterface $eventManager
     * @return TartanManagerInterface
     * @throws CouldNotSaveException
     */
    public function save(TartanManagerInterface $eventManager)
    {
        try {
            $this->resourceEventManager->save($eventManager);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }
        return $eventManager;
    }

    /**
     * Load Block data by given Block Identity
     *
     * @param $eventManagerId
     * @return TartanManagerAlias
     */
    public function getById($eventManagerId)
    {
        /**
         * @var TartanManager $eventManager
         */
        $eventManager = $this->eventManagerFactory->create();
        $this->resourceEventManager->load($eventManager, $eventManagerId);
        return $eventManager;
    }

    /**
     * Load Page data collection by given search criteria
     *
     * @param SearchCriteriaInterface $criteria
     * @return TartanManagerSearchResultsInterface
     */

    public function getList(SearchCriteriaInterface $criteria)
    {
        /** @var Collection $collection */
        $collection = $this->eventManagerCollectionFactory->create();

        foreach ($criteria->getFilterGroups() as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                $condition = $filter->getConditionType() ?: 'eq';
                $collection->addFieldToFilter($filter->getField(), [$condition => $filter->getValue()]);
            }
        }

        $this->collectionProcessor->process($criteria, $collection);

        /** @var TartanManagerSearchResultsInterface $searchResults */
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());
        return $searchResults;
    }

    /**
     * Delete Block
     *
     * @param TartanManagerInterface $eventManager
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function delete(TartanManagerInterface $eventManager)
    {
        try {
            $this->resourceEventManager->delete($eventManager);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * Delete TartanManager by given Block Identity
     *
     * @param string $eventManagerId
     * @return bool|mixed
     */
    public function deleteById($eventManagerId)
    {
        try {
            return $this->delete($this->getById($eventManagerId));
        } catch (\Exception $exception) {
            $this->logger->debug($exception->getMessage());
        }
    }
}
