<?php
namespace Mtc\Finder\ViewModel;

use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Filesystem\Directory\WriteInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Swatches\Helper\Media;
use Mtc\Finder\Api\TartanManagerRepositoryInterface;
use Psr\Log\LoggerInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\UrlInterface;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Filesystem;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Api\FilterBuilder;
use Magento\Catalog\Model\CategoryRepository;
use Magento\Framework\App\Request\Http;
use Mtc\Finder\Model\TartanManagerFactory;
use Magento\Framework\Filesystem\Io\File as IoFile;

/**
 * All tartans view model
 */
class TartansList implements ArgumentInterface
{
    /**
     * @var TartanManagerRepositoryInterface
     */
    private $tartanManagerRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var File
     */
    private $fileDriver;

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * @var FilterBuilder
     */
    private $filterBuilder;

    /**
     * @var CategoryRepository
     */
    private $categoryRepository;

    /**
     * @var Http
     */
    private $request;

    /**
     * @var SortOrderBuilder
     */
    private $sortOrderBuilder;

    /**
     * @var TartanManagerFactory
     */
    private $tartanManager;

    /**
     * @var IoFile
     */
    private $fileIo;

    /**
     * @var WriteInterface
     */
    private $mediaDirectory;

    /**
     * @var Media
     */
    private $swatchMedia;

    /**
     * @param TartanManagerRepositoryInterface $tartanManagerRepository
     * @param LoggerInterface $logger
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param StoreManagerInterface $storeManager
     * @param File $fileDriver
     * @param Filesystem $filesystem
     * @param FilterBuilder $filterBuilder
     * @param CategoryRepository $categoryRepository
     * @param Http $request
     * @param SortOrderBuilder $sortOrderBuilder
     * @param TartanManagerFactory $tartanManager
     * @param IoFile $fileIo
     * @param Media $swatchMedia
     * @throws FileSystemException
     */
    public function __construct(
        TartanManagerRepositoryInterface $tartanManagerRepository,
        LoggerInterface $logger,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoreManagerInterface $storeManager,
        File $fileDriver,
        Filesystem $filesystem,
        FilterBuilder $filterBuilder,
        CategoryRepository $categoryRepository,
        Http $request,
        SortOrderBuilder $sortOrderBuilder,
        TartanManagerFactory $tartanManager,
        IoFile $fileIo,
        Media $swatchMedia
    ) {
        $this->tartanManagerRepository = $tartanManagerRepository;
        $this->logger = $logger;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storeManager = $storeManager;
        $this->fileDriver = $fileDriver;
        $this->filesystem = $filesystem;
        $this->filterBuilder = $filterBuilder;
        $this->categoryRepository = $categoryRepository;
        $this->request = $request;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->tartanManager = $tartanManager;
        $this->fileIo = $fileIo;
        $this->mediaDirectory = $filesystem->getDirectoryWrite(DirectoryList::MEDIA);
        $this->swatchMedia = $swatchMedia;
    }

    /**
     * @return TartanManagerRepositoryInterface
     */
    public function getAllTartans() {
        $search = $this->request->getParam('search');
        $query = $this->request->getParam('query');
        $type = $this->request->getParam('type');
        if ($query and is_numeric($query)) {
            $filters[] = $this->filterBuilder
                ->setField('tartan_id')
                ->setConditionType('eq')
                ->setValue($query)
                ->create();
            $filters[] = $this->filterBuilder
                ->setField('ref_pic')
                ->setConditionType('eq')
                ->setValue(1)
                ->create();
            $this->searchCriteriaBuilder->addFilters($filters);
            $searchCriteria = $this->searchCriteriaBuilder->create();
            return $this->tartanManagerRepository->getList($searchCriteria);
        }
        if ($search or $query) {
            $filters[] = $this->filterBuilder
                ->setField('family')
                ->setConditionType('like')
                ->setValue($search ? $search.'%' : $query.'%')
                ->create();
            $filters[] = $this->filterBuilder
                ->setField('ref_pic')
                ->setConditionType('eq')
                ->setValue(1)
                ->create();
            $sortOrder = $this->sortOrderBuilder->setField('name')->setDirection('ASC')->create();
            $this->searchCriteriaBuilder->addFilters($filters)->setSortOrders([$sortOrder]);
            $searchCriteria = $this->searchCriteriaBuilder->create();
            return $this->tartanManagerRepository->getList($searchCriteria);
        }
        if ($type) {
            $filters[] = $this->filterBuilder
                ->setField('classification')
                ->setConditionType('eq')
                ->setValue($type)
                ->create();
            $filters[] = $this->filterBuilder
                ->setField('ref_pic')
                ->setConditionType('eq')
                ->setValue(1)
                ->create();
            $sortOrder = $this->sortOrderBuilder->setField('name')->setDirection('ASC')->create();
            $this->searchCriteriaBuilder->addFilters($filters)->setSortOrders([$sortOrder]);
            $searchCriteria = $this->searchCriteriaBuilder->create();
            return $this->tartanManagerRepository->getList($searchCriteria);
        }
        return null;
    }

    /**
     * @param $family
     * @param $tartanId
     * @return mixed
     */
    public function getOtherTartans($family, $tartanId) {
        $filters[] = $this->filterBuilder
            ->setField('family')
            ->setConditionType('eq')
            ->setValue($family)
            ->create();
        $filters[] = $this->filterBuilder
            ->setField('ref_pic')
            ->setConditionType('eq')
            ->setValue(1)
            ->create();
        $filters[] = $this->filterBuilder
            ->setField('entity_id')
            ->setConditionType('neq')
            ->setValue($tartanId)
            ->create();
        $filters[] = $this->filterBuilder
            ->setField('sort_order')
            ->setConditionType('eq')
            ->setValue(0)
            ->create();
        $sortOrder = $this->sortOrderBuilder->setField('sort_order')->setDirection('ASC')->create();
        $this->searchCriteriaBuilder->addFilters($filters)->setSortOrders([$sortOrder]);
        $searchCriteria = $this->searchCriteriaBuilder->create();
        $tartans = $this->tartanManagerRepository->getList($searchCriteria);
        if ($tartans->getTotalCount()) {
            $filters = [];
            $filters[] = $this->filterBuilder
                ->setField('family')
                ->setConditionType('eq')
                ->setValue($family)
                ->create();
            $filters[] = $this->filterBuilder
                ->setField('ref_pic')
                ->setConditionType('eq')
                ->setValue(1)
                ->create();
            $filters[] = $this->filterBuilder
                ->setField('entity_id')
                ->setConditionType('neq')
                ->setValue($tartanId)
                ->create();
            $sortOrder = $this->sortOrderBuilder->setField('name')->setDirection('ASC')->create();
            $this->searchCriteriaBuilder->addFilters($filters)->setSortOrders([$sortOrder]);
            $searchCriteria = $this->searchCriteriaBuilder->create();
            return $this->tartanManagerRepository->getList($searchCriteria);
        } else {
            $filters = [];
            $filters[] = $this->filterBuilder
                ->setField('family')
                ->setConditionType('eq')
                ->setValue($family)
                ->create();
            $filters[] = $this->filterBuilder
                ->setField('ref_pic')
                ->setConditionType('eq')
                ->setValue(1)
                ->create();
            $filters[] = $this->filterBuilder
                ->setField('entity_id')
                ->setConditionType('neq')
                ->setValue($tartanId)
                ->create();
            $sortOrder = $this->sortOrderBuilder->setField('sort_order')->setDirection('ASC')->create();
            $this->searchCriteriaBuilder->addFilters($filters)->setSortOrders([$sortOrder]);
            $searchCriteria = $this->searchCriteriaBuilder->create();
            return $this->tartanManagerRepository->getList($searchCriteria);
        }
    }

    /**
     * @return mixed
     */
    public function getSearchParam() {
        return $this->request->getParam('search');
    }

    /**
     * @return mixed
     */
    public function getQueryParam() {
        return $this->request->getParam('query');
    }

    /**
     * @return mixed
     */
    public function getTypeParam() {
        return $this->request->getParam('type');
    }

    /**
     * @return mixed|null
     */
    public function getTartan() {
        $tartanId = $this->request->getParam('tartan_code');
        if($tartanId) {
            try{
                return $this->tartanManagerRepository->getById($tartanId);
            } catch (\Exception $e) {
                $this->logger->critical($e->getMessage());
            }
        }
        return null;
    }

    /**
     * @param $categoryId
     * @param $tartanId
     * @return string|null
     */
    public function getCategoryUrl($categoryId, $tartanId) {
        try {
            $params = ['tartan_code' => $tartanId];
            $urlParams = [];
            foreach ($params as $name=>$value){
                $urlParams[] = $name.'='.urlencode($value);
            }
            $urlParams = implode('&', $urlParams);
            $categoryUrl = $this->categoryRepository->get($categoryId, $this->storeManager->getStore()->getId())->getUrl();
            if ($urlParams){
                $glue = '?';
                if (str_contains($categoryUrl, $glue)){
                    $glue = '&';
                }
                $categoryUrl .= $glue.$urlParams;
            }
            return $categoryUrl;
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }
        return null;
    }

    /**
     * It will return the thumbanil image URL
     *
     * @return string
     * @throws FileSystemException
     */
    public function getTartanImageUrl($file)
    {
        if ($file == '') {
            return '';
        }
        $mediapath = $this->filesystem->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath();
        if (!$this->fileDriver->isExists($mediapath.'/'.$file)) {
            $file = 'TartanManager/awaiting-image.jpg';
        }
        return $this->getMediaUrl($file);
    }

    /**
     * It will return the swatch thumbanil image URL
     *
     * @return string
     * @throws FileSystemException
     */
    public function getTartanSwatchImageUrl($file)
    {
        if ($file == '') {
            return '';
        }
        $mediapath = $this->filesystem->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath();
        $file = str_replace('TartanManager','tartans_images', $file);
        if (!$this->fileDriver->isExists($mediapath.'/attribute/swatch/'.$file)) {
            $file = 'awaiting-image.jpg';
        }
        return $this->getSwatchMediaUrl($file);
    }

    /**
     * Get file url
     * @param $file
     * @return string
     */
    private function getSwatchMediaUrl($file)
    {
        try {
            $file = ltrim(str_replace('\\', '/', $file), '/');
            return $this->storeManager
                    ->getStore()
                    ->getBaseUrl(UrlInterface::URL_TYPE_MEDIA) . 'attribute/swatch/' .$file;
        } catch (\Exception $exception) {
            $this->logger->critical($exception->getMessage());
        }
    }

    /**
     * Get file url
     * @param $file
     * @return string
     */
    private function getMediaUrl($file)
    {
        try {
            $file = ltrim(str_replace('\\', '/', $file), '/');
            return $this->storeManager
                    ->getStore()
                    ->getBaseUrl(UrlInterface::URL_TYPE_MEDIA) . $file;
        } catch (\Exception $exception) {
            $this->logger->critical($exception->getMessage());
        }
    }

    /**
     * @param $entityId
     * @param $tartanId
     * @param $family
     * @return array|null
     */
    public function getRelatedTartans($entityId, $tartanId, $family)
    {
        $filters = [];
        $filters[] = $this->filterBuilder
            ->setField('family')
            ->setConditionType('eq')
            ->setValue($family)
            ->create();
        $filters[] = $this->filterBuilder
            ->setField('ref_pic')
            ->setConditionType('eq')
            ->setValue(0)
            ->create();
        $filters[] = $this->filterBuilder
            ->setField('tartan_id')
            ->setConditionType('eq')
            ->setValue($tartanId)
            ->create();
        $filters[] = $this->filterBuilder
            ->setField('entity_id')
            ->setConditionType('neq')
            ->setValue($entityId)
            ->create();
        $sortOrder = $this->sortOrderBuilder->setField('name')->setDirection('ASC')->create();
        $this->searchCriteriaBuilder->addFilters($filters)->setSortOrders([$sortOrder]);
        $searchCriteria = $this->searchCriteriaBuilder->create();
        return $this->tartanManagerRepository->getList($searchCriteria);
    }

    /**
     * @param $imgValue
     * @param $color
     * @return string
     * @throws FileSystemException
     * @throws NoSuchEntityException
     */
    private function getTartanColorPath($imgValue, $color) {
        $pathInfo = $this->fileIo->getPathInfo($imgValue);
        $filename = $pathInfo['filename'];
        $filename = $filename.$color.'.';
        $tartanPath = '/tartans_images'.'/'.$filename.$pathInfo['extension'];
        $mediaPath = $this->swatchMedia->getSwatchMediaPath().$tartanPath;
        $colorImgPath = $this->mediaDirectory
            ->getAbsolutePath($this->swatchMedia->getSwatchMediaPath(). $tartanPath);
        if ($this->fileDriver->isExists($colorImgPath)) {
            $mediaUrl = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
            return $mediaUrl.$mediaPath;
        }
        return '';
    }
}
