<?php
namespace Mtc\InteractiveWeaver\Controller\Weaver;

use Magento\Contact\Model\ConfigInterface;
use Magento\Framework\App\Action\HttpPostActionInterface as HttpPostActionInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Controller\Result\RedirectFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Translate\Inline\StateInterface;
use Magento\Framework\Message\ManagerInterface;
use Mtc\InteractiveWeaver\Model\Mail\TransportBuilder;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Data\Form\FormKey\Validator;
use \Magento\Framework\Exception\NotFoundException;

class Submit implements HttpPostActionInterface
{
    const XML_PATH_EMAIL_TEMPLATE = 'design_submission_email_template';

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var RedirectFactory
     */
    private $resultRedirectFactory;

    /**
     * @var StateInterface
     */
    private $inlineTranslation;

    /**
     * @var ManagerInterface
     */
    private $messageManager;

    /**
     * @var TransportBuilder
     */
    private $transportBuilder;

    /**
     * @var ConfigInterface
     */
    private $contactsConfig;

    /**
     * @var File
     */
    private $fileDriver;

    /**
     * @var Validator
     */
    private $formValidator;

    /**
     * @param RequestInterface $request
     * @param RedirectFactory $resultRedirectFactory
     * @param StateInterface $inlineTranslation
     * @param ManagerInterface $messageManager
     * @param TransportBuilder $transportBuilder
     * @param ConfigInterface $contactsConfig
     * @param File $fileDriver
     * @param Validator $formValidator
     */
    public function __construct(
        RequestInterface $request,
        RedirectFactory $resultRedirectFactory,
        StateInterface $inlineTranslation,
        ManagerInterface $messageManager,
        TransportBuilder $transportBuilder,
        ConfigInterface $contactsConfig,
        File $fileDriver,
        Validator $formValidator
    ) {
        $this->request = $request;
        $this->resultRedirectFactory = $resultRedirectFactory;
        $this->inlineTranslation = $inlineTranslation;
        $this->messageManager = $messageManager;
        $this->transportBuilder = $transportBuilder;
        $this->contactsConfig = $contactsConfig;
        $this->fileDriver = $fileDriver;
        $this->formValidator = $formValidator;
    }

    /**
     * @return ResponseInterface|Redirect|ResultInterface
     * @throws NotFoundException
     */
    public function execute()
    {
        if (!$this->request->isPost()) {
            return $this->resultRedirectFactory->create()->setPath('*/*/');
        }
        if (!$this->formValidator->validate($this->request)) {
            throw new NotFoundException(__('Page Not Found.'));
        }
        try{
            $postData = $this->request->getPostValue();
            $filesData = $this->request->getFiles('document');
            if ($postData['pattern'] == '' && $filesData['name'] == '') {
                $this->messageManager->addErrorMessage(
                    __('Please design a tartan or upload custom design before submission.')
                );
                return $this->resultRedirectFactory->create()->setPath('*/*/');
            }
            $this->sendEmail();
            $this->messageManager->addSuccess(
                __('Thanks for sending us your design. We\'ll respond to you very soon.')
            );
            return $this->resultRedirectFactory->create()->setPath('*/*/');
        } catch (\Exception $e) {
            $this->inlineTranslation->resume();
            $this->messageManager->addErrorMessage(
                __('We can\'t process your request right now. Sorry, that\'s all we know.')
            );
            return $this->resultRedirectFactory->create()->setPath('*/*/');
        }
    }

    /**
     * @return void
     * @throws FileSystemException
     * @throws LocalizedException
     * @throws MailException
     */
    private function sendEmail()
    {
        $filesData = $this->request->getFiles('document');
        $this->inlineTranslation->suspend();
        $postObject = new \Magento\Framework\DataObject();
        $post = $this->request->getPostValue();
        $post['interested_in'] = '';
        if (isset($post['mediumweight'])) {
            $post['interested_in'] = $post['mediumweight'];
        }
        if (isset($post['heavyweight'])) {
            $post['interested_in'] .= ' ' . $post['heavyweight'];
        }
        if (isset($post['lightweight'])) {
            $post['interested_in'] .= ' ' . $post['lightweight'];
        }
        if (isset($post['sampleprint'])) {
            $post['interested_in'] .= ' ' . $post['sampleprint'];
        }
        $postObject->setData($post);
        if ($filesData['name']) {
            $transport = $this->transportBuilder
                ->setTemplateIdentifier(self::XML_PATH_EMAIL_TEMPLATE)
                ->setTemplateOptions(
                    [
                        'area' => \Magento\Framework\App\Area::AREA_FRONTEND,
                        'store' => \Magento\Store\Model\Store::DEFAULT_STORE_ID,
                    ]
                )
                ->setTemplateVars(['data' => $postObject])
                ->addTo($this->contactsConfig->emailRecipient())
                ->setReplyTo($post['email'], $post['firstname'] . ' ' . $post['lastname'])
                ->setFrom($this->contactsConfig->emailSender())
                ->addAttachment($this->fileDriver->fileGetContents($filesData['tmp_name']), $filesData['name'], $filesData['type'])
                ->getTransport();
        } else {
            $transport = $this->transportBuilder
                ->setTemplateIdentifier(self::XML_PATH_EMAIL_TEMPLATE)
                ->setTemplateOptions(
                    [
                        'area' => \Magento\Framework\App\Area::AREA_FRONTEND,
                        'store' => \Magento\Store\Model\Store::DEFAULT_STORE_ID,
                    ]
                )
                ->setTemplateVars(['data' => $postObject])
                ->addTo($this->contactsConfig->emailRecipient())
                ->setReplyTo($post['email'], $post['firstname'] . ' ' . $post['lastname'])
                ->setFrom($this->contactsConfig->emailSender())
                ->getTransport();
        }
        $transport->sendMessage();
        $this->inlineTranslation->resume();
    }

    /**
     * @return array
     * @throws LocalizedException
     */
    private function validatedParams()
    {
        $request = $this->request;
        if (trim($request->getParam('tartan-name', '')) === '') {
            throw new LocalizedException(__('Enter the Tartan Name and try again.'));
        }
        if (trim($request->getParam('firstname', '')) === '') {
            throw new LocalizedException(__('Enter the First Name and try again.'));
        }
        if (trim($request->getParam('lastname', '')) === '') {
            throw new LocalizedException(__('Enter the Last Name and try again.'));
        }
        if (\strpos($request->getParam('email', ''), '@') === false) {
            throw new LocalizedException(__('The email address is invalid. Verify the email address and try again.'));
        }
        return $request->getParams();
    }
}
