<?php

namespace Mtc\MembersBasket\Http\Controllers;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Members\HasMembersNavigation;
use Mtc\Members\Member;
use Mtc\MembersBasket\Http\Requests\MembersOrderRequest;
use Mtc\Orders\Contracts\OrderContract;

/**
 * Class OrderController
 *
 * @package Mtc\Modules\Members\Classes\Http\Controllers
 */
class OrderController extends Controller
{
    use HasMembersNavigation;

    /**
     * View order list
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        /** @var Member $member */
        $member = MemberAuth::user();
        $orders = App::Make(OrderContract::class)->newQuery()
            ->with('items')
            ->paid()
            ->where(function (Builder $query) use ($member) {
                $query->where('member_id', $member->id)
                    ->orWhere('email', $member->email);
            })
            ->latest()
            ->get()
            ->map(function ($order) {
                $order->num_items = $order->items->sum('quantity');
                return $order;
            });

        $this->page_meta['title'] = 'My Past Orders | ' . config('app.name');
        $this->page_meta['page_title'] = 'My Orders';
        return template('/members/orders/index.twig', [
            'page_meta' => $this->page_meta,
            'member_id' => $member->id,
            'orders' => $orders,
            'sections' => $this->sections(),
        ]);

    }

    /**
     * View Single order
     *
     * @param MembersOrderRequest $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     * @throws \Exception
     */
    public function show(MembersOrderRequest $request)
    {
        $items = $request->order()->items;
        $totals = [
            'subtotal' => $request->order()->cost_subtotal->withTax(),
            'total' => $request->order()->cost_total->withTax(),
            'delivery' => optional($request->order()->deliverySurcharge->surcharge_amount)->withTax()
        ];

        $this->page_meta['title'] = 'Order Details | ' . config('app.name');
        $this->page_meta['page_title'] = 'Order #' . $request->order()->id;
        return template('/members/orders/show.twig', [
            'page_meta' => $this->page_meta,
            'member_id' => MemberAuth::id(),
            'order' => $request->order(),
            'totals' => $totals,
            'items' => $items,
            'sections' => $this->sections(),
        ]);
    }
}
