<?php

namespace Mtc\MembersBasket;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Mtc\Checkout\Invoice;
use Mtc\Foundation\Address;
use Mtc\Members\Member;
use Mtc\MembersBasket\Exceptions\MemberAlreadyExists;
use Mtc\Orders\Contracts\OrderContract;

/**
 * Class PaymentRegistrationMemberFactory
 *
 * @package Mtc\MembersBasket
 */
class PaymentRegistrationMemberFactory
{

    /**
     * Create members account
     *
     * @param Invoice $invoice
     * @param Request $request
     * @throws MemberAlreadyExists
     * @return Member
     */
    public function create(Invoice $invoice, Request $request)
    {
        switch ($invoice->payable_type) {
            default:
            case "order":
                return $this->createFromOrder($invoice->payable, $request);
                break;
        }
    }

    /**
     * Generate member from order data
     *
     * @param OrderContract $invoice
     * @throws MemberAlreadyExists
     */
    protected function createFromOrder(OrderContract $order, Request $request)
    {
        $exists = Member::query()
            ->where('email', $order->email)
            ->exists();

        if ($exists) {
            throw new MemberAlreadyExists();
        }

        $member = new Member();
        $member->fill([
            'name' => $order->customer_name,
            'email' => $order->email,
            'contact_no' => $order->contact_number,
            'password' => Hash::make($request->input('password')),
        ]);
        $member->save();

        if ($order->billingAddress) {
            $billing_fields = $order->billingAddress->only(array_keys(Address::$blueprint));
            $address = $member->addresses()->create($billing_fields);
        }

        if ($order->shippingAddress) {
            $shipping_fields = $order->shippingAddress->only(array_keys(Address::$blueprint));
            if (!empty($billing_fields) && !empty(array_diff($shipping_fields, $billing_fields))) {
                $address = $member->addresses()->create($shipping_fields);
            }
        }
        if ($address) {
            $member->default_address_id = $address->id;
            $member->save();
        }

        return $member;
    }
}
