<?php

namespace Mtc\Members\Http\Controllers\Auth;

use Illuminate\Auth\Passwords\DatabaseTokenRepository;
use Illuminate\Foundation\Application;
use Illuminate\Foundation\Auth\SendsPasswordResetEmails;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Mtc\Core\Admin\User;
use Mtc\Members\Contracts\MemberModel;
use Mtc\Members\Member;

class ForgotPasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset emails and
    | includes a trait which assists in sending these notifications from
    | your application to your users. Feel free to explore this trait.
    |
    */


    /**
     * @var Application
     */
    protected $app;

    /**
     * Render the password reset form
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show()
    {
        $this->page_meta['title'] = 'Forgot Password | ' . config('app.name');
        $this->page_meta['page_title'] = 'Forgot Your Password?';
        return template('members/forgot_password.twig', [
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Handle password reset
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function send(Request $request, MemberModel $member)
    {
        $request->validate([
            'email' => ['required', 'email'],
        ]);

        // We will send the password reset link to this user. Once we have attempted
        // to send the link, we will examine the response then see the message we
        // need to show to the user. Finally, we'll send out a proper response.
        $status = Password::sendResetLink(
            $request->only('email')
        );

        return $status == Password::RESET_LINK_SENT
            ? back()->with('status', __($status))
            : back()->withInput($request->only('email'))
                ->withErrors(['email' => __($status)]);
    }

    /**
     * Send a reset link to the given user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return void
     */
    public function sendResetLinkEmail(Request $request, MemberModel $member)
    {
        /** @var User $user */
        $member = $member->newQuery()->where('email', $request->input('email'))->first();
        if (!$member) {
            return;
        }

        $this->app = app();
        $config = $this->app['config']["auth.passwords.members"];
        $token_repository = $this->createTokenRepository($config);
        $token = $token_repository->create($member);

        $this->sendEmail($member, $token);
    }

    /**
     * Create token repository
     *
     * @param array $config
     * @return DatabaseTokenRepository
     */
    protected function createTokenRepository(array $config)
    {
        $key = $this->app['config']['app.key'];

        if (Str::startsWith($key, 'base64:')) {
            $key = base64_decode(substr($key, 7));
        }

        $connection = $config['connection'] ?? null;

        return new DatabaseTokenRepository(
            $this->app['db']->connection($connection),
            $this->app['hash'],
            $config['table'],
            $key,
            $config['expire']
        );
    }

    /**
     * Send the email to user
     *
     * @param MemberModel $member
     * @param $token
     */
    protected function sendEmail(MemberModel $member, $token)
    {
        $subject = 'Password Reset on ' . config('app.name');
        $content = template('members/emails/reset_password.twig', [
            'url' => URL::signedRoute('members.password.show_reset', [
                'token' => $token,
                'email' => $member->email,
            ])
        ]);
        email($member->email, $subject, $content);
    }
}
